#pragma once

#include "filter.h"
#include "paging.h"
#include "time_range.h"

#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/serialization/json.h"

namespace communication {
namespace command {

//----------------------------- Список команд --------------------------------


/** WEB
  Команда может редактировать только описание оценки
*/
extern const QUuidEx ScoreEdit;

/** WEB
*/
extern const QUuidEx ScoreDelete;

/** WEB
*/
extern const QUuidEx ScoreInfo;

/** WEB
*/
extern const QUuidEx ScoreList;

} // namespace command

// Статус экспорта оценки в ФОМС
enum class SendScoreStatus : quint32
{
    Undefined = 0,
    Created   = 1, // Оценка создана и не отправлена
    NeedSend  = 2, // Необходимо экспортировать оценку в ФОМС
    IsSent    = 3  // Оценка уже экспортирована в ФОМС
};

//---------------- Структуры данных используемые в сообщениях ----------------

namespace data {

struct Score
{
    // Идентификатор оценки
    QUuidEx id;

    // Идентификатор задачи
    QUuidEx taskId;

    // Идентификатор пользователя
    QUuidEx userId;

    // Идентификатор модели
    QUuidEx modelId;

    // Наименование оценки
    QString name;

    // Описание оценки
    QString description;

    // Время создания оценки
    QDateTime createDate = {QDateTime::currentDateTime()};

    // Левая и правая граница интервала выборки данных для оценки
    TimeRange period;

    // TODO удалить
    // Фактическая левая и правая граница интервала выборки данных для оценки
    TimeRange periodReal;

    // Признак определяет, что оценка будет доступна всем пользователям
    bool isPublic = {false};

    // Признак архивной оценки, если равно TRUE, то оценка не будет отображается
    // в интерфейсе пользователя
    bool isArchive = {false};

    // Количество строк, изменившихся в процессе синхронизации после
    // создания оценки
    qint32 rowsOutdated = {0};

    // Состояние статуса экспорта оценки
    SendScoreStatus sendStatus = {SendScoreStatus::Undefined};

    //--- Вспомогательные поля для WEB-интерфейса ---
    QString taskName;
    QString modelName;

    // Признак определяет, что оценка была создана периодической задачей
    bool isPeriodic = {false};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id          )
        J_SERIALIZE_ITEM( taskId      )
        J_SERIALIZE_ITEM( userId      )
        J_SERIALIZE_ITEM( modelId     )
        J_SERIALIZE_ITEM( name        )
        J_SERIALIZE_ITEM( description )
        J_SERIALIZE_ITEM( createDate  )
        J_SERIALIZE_ITEM( period      )
        J_SERIALIZE_OPT ( periodReal  )
        J_SERIALIZE_ITEM( isPublic    )
        J_SERIALIZE_OPT ( isArchive   )
        J_SERIALIZE_ITEM( sendStatus  )
        J_SERIALIZE_ITEM( taskName    )
        J_SERIALIZE_ITEM( modelName   )
        J_SERIALIZE_OPT ( isPeriodic  )
    J_SERIALIZE_END
};

struct ScoreFilter
{
    // Идентификатор оценки
    QUuidEx id;

    // Идентификатор задачи
    QUuidEx taskId;

    // Идентификатор пользователя (владельца оценки)
    QUuidEx userId;

    // Полное или частичное имя оценки
    QString name;

    // Отображать публичные оценки других пользователей
    filter::Show show = {filter::Show::Self};

    // Отображать архивные оценки (по умолчанию - не отображать)
    filter::Archive archive = {filter::Archive::No};

    // Идентификаторы дополнительной фильтрации по пользователям, используются
    // совместно с filter::Show
    QVector<QUuidEx> otherUsers;

    // Фильтр "Дата создания"
    TimeRange createDate;

    // Фильтр "Период данных"
    TimeRange dataPeriod;

    // Идентификатор модели, которая использовалась при создании оценки
    QUuidEx modelId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id         )
        J_SERIALIZE_ITEM( taskId     )
        J_SERIALIZE_ITEM( userId     )
        J_SERIALIZE_ITEM( name       )
        J_SERIALIZE_ITEM( show       )
        J_SERIALIZE_OPT ( archive    )
        J_SERIALIZE_ITEM( otherUsers )
        J_SERIALIZE_ITEM( createDate )
        J_SERIALIZE_ITEM( dataPeriod )
        J_SERIALIZE_ITEM( modelId    )
    J_SERIALIZE_END
};

struct ScoreEdit : Data<&command::ScoreEdit,
                         Message::Type::Command,
                         Message::Type::Answer>
{
    QUuidEx id;
    QUuidEx userId;
    QString name;
    QString description;
    bool    isPublic = {false};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id          )
        J_SERIALIZE_ITEM( userId      )
        J_SERIALIZE_ITEM( name        )
        J_SERIALIZE_ITEM( description )
        J_SERIALIZE_ITEM( isPublic    )
    J_SERIALIZE_END
};

struct ScoreDelete : Data<&command::ScoreDelete,
                           Message::Type::Command,
                           Message::Type::Answer>
{
    QUuidEx id;
    QUuidEx userId;

    // Признак принудительного удаления оценки, если равен TRUE, то оценка
    // будет удалена даже если у нее есть зависимости
    bool force = {false};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id     )
        J_SERIALIZE_ITEM( userId )
        J_SERIALIZE_OPT ( force  )
    J_SERIALIZE_END
};

struct ScoreInfo : Data<&command::ScoreInfo,
                         Message::Type::Command>
{
    QUuidEx id;
    QUuidEx userId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id     )
        J_SERIALIZE_ITEM( userId )
    J_SERIALIZE_END
};

struct ScoreInfoA : Score, Data<&command::ScoreInfo,
                                 Message::Type::Answer>
{};

struct ScoreList : Data<&command::ScoreList,
                         Message::Type::Command,
                         Message::Type::Answer>
{
    // Фильтр выбора данных
    ScoreFilter filter;

    // Фильтр для постраничной выборки данных
    PagingInfo paging;

    // Результирующая выборка
    QVector<Score> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( filter )
        J_SERIALIZE_ITEM( paging )
        J_SERIALIZE_ITEM( items  )
    J_SERIALIZE_END
};

} // namespace data
} // namespace communication


