
#pragma once

#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/serialization/json.h"

namespace communication {
namespace command {

//----------------------------- Список команд --------------------------------

/** WEB
  Команда используется для получения статуса о состоянии учетной записи пользо-
  вателя: активна/не активна
*/
extern const QUuidEx UserActiveStatus;

/** WEB
  Команда устанавливает флаг активности пользователя, команда отправляется адми-
  нистратором AisExpert из Web-интерфейса. Положительный ответ содержит значение
  записанное в БД. В случае возникновения ошибки актуальный флаг активности
  пользователя можно получить при помощи команды UserActiveStatus
*/
extern const QUuidEx UserActiveSet;

/** WEB
  Команда отправляется на Web-сервер с целью аннулировать все сессии  пользова-
  теля. Команда отправляется когда учетная запись пользователя отключается или
  когда пользователь исключен из групп AisExpert/AisExpertAdmin в Windows
  Active Directory.
  Поле Message::tag содержит hashId-пользователя
*/
extern const QUuidEx UserResetAuth;

/** WEB
  Команда отправляется из web-интерфейса что бы уведомить AisExpert о том, что
  пользователь вышел из своего кабинета.
  Поле Message::tag содержит hashId-пользователя
*/
extern const QUuidEx UserLogout;

/** WEB
  Получение данных по конкретному пользователю по его id
*/
extern const QUuidEx UserInfo;

/** WEB
    Вывод списка пользователей из БД АИС
*/
extern const QUuidEx UserList;

} // namespace command

//---------------- Структуры данных используемые в сообщениях ----------------

namespace data {

struct UserActiveStatus : Data<&command::UserActiveStatus,
                                Message::Type::Command,
                                Message::Type::Answer>
{
    QUuidEx id;          // id пользователя
    bool value = {true}; // Значение статуса

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id )
        J_SERIALIZE_ITEM( value  )
    J_SERIALIZE_END
};

struct UserActiveSet : Data<&command::UserActiveSet,
                             Message::Type::Command,
                             Message::Type::Answer>
{
    QUuidEx id;          // id пользователя для которого меняется статус
    bool value = {true}; // Значение статуса

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id )
        J_SERIALIZE_ITEM( value  )
    J_SERIALIZE_END
};

struct User
{
    QUuidEx   id;
    quint64   hashId = {0};
    QString   login;
    QString   name;
    bool      isActive = {false};
    bool      isValid  = {false};
    bool      isAdmin  = {false};
    QUuidEx   groupId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id       )
        J_SERIALIZE_ITEM( hashId   )
        J_SERIALIZE_ITEM( login    )
        J_SERIALIZE_ITEM( name     )
        J_SERIALIZE_ITEM( isActive )
        J_SERIALIZE_ITEM( isValid  )
        J_SERIALIZE_ITEM( isAdmin  )
        J_SERIALIZE_ITEM( groupId  )
    J_SERIALIZE_END
};

struct UserInfo : Data<&command::UserInfo,
                        Message::Type::Command>
{
    QUuidEx id; // id пользователя, по которому запрашивается информация
    J_SERIALIZE_ONE( id )
};

struct UserInfoA : User, Data<&command::UserInfo,
                               Message::Type::Answer>
{};

struct UserList : Data<&command::UserList,
                        Message::Type::Command,
                        Message::Type::Answer>
{
    QString filterByLogin;
    QVector<User> items;
    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( filterByLogin )
        J_SERIALIZE_ITEM( items         )
    J_SERIALIZE_END
};

} // namespace data
} // namespace communication


