#include "main_window.h"
#include "ui_main_window.h"
#include "functions.h"

#include "commands/authorization.h"
#include "commands/commands.h"
#include "commands/sync_data.h"
#include "commands/nsi.h"

#include "shared/defmac.h"
#include "shared/logger/logger.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/config/config.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/logger_operators.h"

#include <QMessageBox>
#include <unistd.h>

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "Application")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "Application")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "Application")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "Application")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "Application")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "Application")

#define QTEXT(MSG) QString::fromUtf8(MSG)

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    #define FUNC_REGISTRATION(COMMAND) \
        _funcInvoker.registration(command:: COMMAND, &MainWindow::command_##COMMAND, this);

    FUNC_REGISTRATION(FomsAuthorization)
    FUNC_REGISTRATION(KeepWaitCommand)
    FUNC_REGISTRATION(GetSyncData)
    FUNC_REGISTRATION(SyncDataCheck)
    FUNC_REGISTRATION(SyncNsiVidmp)

    #undef FUNC_REGISTRATION

    _labelConnectStatus = new QLabel(tr("Disconnected"), this);
    ui->statusBar->addWidget(_labelConnectStatus);
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::timerEvent(QTimerEvent* event)
{
    if (event->timerId() == _fomsConTimerId)
    {
        checkFomsConnect();
    }
}

bool MainWindow::init()
{
    chk_connect_q(fomsCon().socket().get(), SIGNAL(message(communication::Message::Ptr)),
                  this, SLOT(message(communication::Message::Ptr)))

    chk_connect_q(fomsCon().socket().get(), SIGNAL(connected(communication::SocketDescriptor)),
                  this, SLOT(fomsSocketConnected(communication::SocketDescriptor)))

    chk_connect_q(fomsCon().socket().get(), SIGNAL(disconnected(communication::SocketDescriptor)),
                  this, SLOT(fomsSocketDisconnected(communication::SocketDescriptor)))

    bool autoConnect = true;
    config::state().getValue("startup.auto_connect", autoConnect);
    ui->chkAutoConnect->setChecked(autoConnect);

    bool autoAuthoriz = true;
    config::state().getValue("startup.auto_authoriz", autoAuthoriz);
    ui->chkAutoAuthoriz->setChecked(autoAuthoriz);

    bool syncDataCheck = true;
    config::state().getValue("sync_data.check", syncDataCheck);
    ui->chkSyncDataCheck->setChecked(syncDataCheck);

    return true;
}

void MainWindow::deinit()
{
    killTimer(_fomsConTimerId);
}

void MainWindow::saveGeometry()
{
    QPoint p = pos();
    std::vector<int> v {p.x(), p.y(), width(), height()};
    config::state().setValue("windows.main_window.geometry", v);
}

void MainWindow::loadGeometry()
{
    std::vector<int> v {0, 0, 800, 600};
    config::state().getValue("windows.main_window.geometry", v);
    move(v[0], v[1]);
    resize(v[2], v[3]);
}

void MainWindow::saveSettings()
{
}

void MainWindow::loadSettings()
{
}

void MainWindow::message(const Message::Ptr& message)
{
    if (message->processed())
        return;

    if (lst::FindResult fr = _funcInvoker.findCommand(message->command()))
    {
        if (!command::pool().commandIsMultiproc(message->command()))
            message->markAsProcessed();
        _funcInvoker.call(message, fr);
    }
}

void MainWindow::fomsSocketConnected(SocketDescriptor /*socketDescriptor*/)
{
    log_verbose_m << u8"Соединение c сервером ФОМС установлено";

    bool autoAuthoriz = true;
    config::state().getValue("startup.auto_authoriz", autoAuthoriz);

    if (autoAuthoriz)
        on_btnAuthorization_clicked(true);

    _labelConnectStatus->setText("Connected (Not Authorized)");
}

void MainWindow::fomsSocketDisconnected(SocketDescriptor /*socketDescriptor*/)
{
    fomsCon().setAuthorized(false);
    log_error_m << u8"Соединение c сервером ФОМС потеряно";
    _labelConnectStatus->setText("Disconnected");
}

void MainWindow::checkFomsConnect()
{
    if (_fomsConTimerId == -1)
    {
        int interval = 10; /*в секундах*/
        config::base().getValue("foms.reconnect_timeout", interval);
        _fomsConTimerId = startTimer(interval * 1000);
    }

    bool autoConnect = true;
    config::state().getValue("startup.auto_connect", autoConnect);
    if (!autoConnect)
        return;

    if (!fomsCon().isConnected())
    {
        log_info_m << u8"Попытка установки соединения с сервером ФОМС";
        fomsCon().socket()->connect();
    }
}

void MainWindow::on_btnConnect_clicked(bool)
{
    if (!fomsCon().isConnected())
    {
        log_info_m << u8"Попытка установки соединения с сервером ФОМС";
        fomsCon().socket()->connect();
    }
    else
    {
        QString msg = QTEXT(u8"Соединение уже установлено");
        QMessageBox::information(this, qApp->applicationName(), msg);
    }
}

void MainWindow::on_btnDisconnect_clicked(bool)
{
    if (fomsCon().isConnected())
    {
        fomsCon().disconnect();
        QString msg = QTEXT(u8"Соединение разорвано");
        QMessageBox::information(this, qApp->applicationName(), msg);
    }
}

void MainWindow::on_btnAuthorization_clicked(bool)
{
    QString password;
    config::base().rereadFile();
    config::base().getValue("foms.password", password);

    if (!password.isEmpty())
    {
        data::FomsAuthorization fomsAuthorization;
        fomsAuthorization.password = password;

        Message::Ptr m = createJsonMessage(fomsAuthorization);
        fomsCon().socket()->send(m);
    }
}

void MainWindow::on_chkAutoConnect_clicked(bool)
{
    config::state().setValue("startup.auto_connect", ui->chkAutoConnect->isChecked());
    config::state().save();
}

void MainWindow::on_chkAutoAuthoriz_clicked(bool)
{
    config::state().setValue("startup.auto_authoriz", ui->chkAutoAuthoriz->isChecked());
    config::state().save();
}

void MainWindow::on_chkSyncDataCheck_clicked(bool)
{
    config::state().setValue("sync_data.check", ui->chkSyncDataCheck->isChecked());
    config::state().save();
}

void MainWindow::on_btnGetSyncData1_clicked(bool)
{
    QDate day {2019, 1, 1};
    QDateTime maxTimeMark {QDate(1900, 1, 1)};

    data::GetSyncData getSyncData;
    getSyncData.period.begin = QDateTime(day);
    getSyncData.period.end = QDateTime(day);
    getSyncData.timeMark = maxTimeMark;
    getSyncData.count = 5000;

    Message::Ptr message = createJsonMessage(getSyncData);
    message->setPriority(Message::Priority::Low);

    _waitTimeout = _defaultWaitTimeout - 2;
    message->setMaxTimeLife(_waitTimeout);

    if (!fomsCon().send(message))
    {
        log_error_m << u8"Соединение с ФОМС сервером отсутствует";
        return;
    }

    log_debug2_m << "Message sent"
                 << ". Id: " << message->id()
                 << ". Command: " << CommandNameLog(message->command());
}

void MainWindow::on_btnGetSyncData2_clicked(bool)
{
    QDate day {2019, 1, 1};
    QDateTime timeMark {QDate(1900, 1, 1)};

    data::GetSyncData getSyncData;
    getSyncData.period.begin = QDateTime(day);
    getSyncData.period.end = QDateTime(day);
    getSyncData.timeMark = timeMark;
    getSyncData.count = 5000;

    simple_timer timer;
    data::GetSyncData getSyncDataAnsw;

    // Блок 1
    Message::Ptr message = createJsonMessage(getSyncData);
    message->setPriority(Message::Priority::Low);

    _waitTimeout = _defaultWaitTimeout - 2;
    message->setMaxTimeLife(_waitTimeout);

    // Признак второго примера
    message->setTag(2);

    // Отправляем запрос на первый блок данных
    if (!fomsCon().send(message))
    {
        log_error_m << u8"Соединение с ФОМС сервером отсутствует";
        return;
    }
    log_debug2_m << "Message sent"
                 << ". Id: " << message->id()
                 << ". Command: " << CommandNameLog(message->command());

    timer.reset();
    _answerMessage.reset();

    // Ждем ответ 60 сек.
    while (timer.elapsed() < (_waitTimeout * 1000))
    {
        usleep(10*1000);
        QCoreApplication::processEvents();
        if (!_answerMessage.empty())
            break;
    }

    if (_answerMessage.empty())
    {
        log_error_m << "Message timer elapsed: " << timer.elapsed()
                    << ". Wait timeout: " << _waitTimeout * 1000 << " ms";
        ui->txtGetSyncData2->append(QTEXT(u8"Превышен таймаут"));
        return;
    }

    SResult res = readFromMessage(_answerMessage, getSyncDataAnsw);
    if (!res)
    {
        log_error_m << u8"Ошибка разбора json-выражения полученного от ФОМС сервера";
        return;
    }

    QDateTime maxTimeMark = QDateTime(QDate(1900, 1, 1));

    // Определяем максимальный полученный TIME_MARK
    for (int i = 0; i < getSyncDataAnsw.items.count(); ++i)
        if (getSyncDataAnsw.items[i].TIME_MARK > maxTimeMark)
            maxTimeMark = getSyncDataAnsw.items[i].TIME_MARK;

    ui->txtGetSyncData2->append(QTEXT(u8"Получено записей: ") + QString::number(getSyncDataAnsw.items.count()));
    ui->txtGetSyncData2->append(QTEXT(u8"Максимальный TIME_MARK: ") + maxTimeMark.toString("dd.MM.yyyy hh:mm:ss.zzz"));

    // Блок 2
    // Берем максимальный timeMark от предыдущего блока данных
    getSyncData.timeMark = maxTimeMark;

//    qint64 i1 = maxTimeMark.toMSecsSinceEpoch();
//    qint64 i2 = getSyncData.timeMark.toMSecsSinceEpoch();

    message = createJsonMessage(getSyncData);
    message->setPriority(Message::Priority::Low);

    _waitTimeout = _defaultWaitTimeout - 2;
    message->setMaxTimeLife(_waitTimeout);
    message->setTag(2);

    // Отправляем запрос на второй блок данных
    if (!fomsCon().send(message))
    {
        log_error_m << u8"Соединение с ФОМС сервером отсутствует";
        return;
    }
    log_debug2_m << "Message sent"
                 << ". Id: " << message->id()
                 << ". Command: " << CommandNameLog(message->command());

    timer.reset();
    _answerMessage.reset();

    // Ждем ответ 60 сек.
    while (timer.elapsed() < (_waitTimeout * 1000))
    {
        usleep(10*1000);
        QCoreApplication::processEvents();
        if (!_answerMessage.empty())
            break;
    }

    if (_answerMessage.empty())
    {
        log_error_m << "Message timer elapsed: " << timer.elapsed()
                    << ". Wait timeout: " << _waitTimeout * 1000 << " ms";
        ui->txtGetSyncData2->append(QTEXT(u8"Превышен таймаут"));
        return;
    }

    res = readFromMessage(_answerMessage, getSyncDataAnsw);
    if (!res)
    {
        log_error_m << u8"Ошибка разбора json-выражения полученного от ФОМС сервера";
        return;
    }

    maxTimeMark = QDateTime(QDate(1900, 1, 1));

    // Определяем максимальный полученный TIME_MARK
    for (int i = 0; i < getSyncDataAnsw.items.count(); ++i)
        if (getSyncDataAnsw.items[i].TIME_MARK > maxTimeMark)
            maxTimeMark = getSyncDataAnsw.items[i].TIME_MARK;

    ui->txtGetSyncData2->append(QTEXT(u8"Получено записей: ") + QString::number(getSyncDataAnsw.items.count()));
    ui->txtGetSyncData2->append(QTEXT(u8"Максимальный TIME_MARK: ") + maxTimeMark.toString("dd.MM.yyyy hh:mm:ss.zzz"));
}

struct CrcItem
{
    QUuidEx   GKEY;
    QDateTime TIME_MARK;
};

int crcCompare(const CrcItem* item1, const CrcItem* item2, void* /*extParam*/ = 0)
{
    LIST_COMPARE_MULTI_ITEM(item1->GKEY, item2->GKEY)
    LIST_COMPARE_MULTI_ITEM(item1->TIME_MARK, item2->TIME_MARK)
    return 0;
}

void MainWindow::on_btnGetSyncData3_clicked(bool)
{
    data::GetSyncData getSyncData;
    getSyncData.period.begin = QDateTime(QDate::fromString(ui->linePeriodBegin->text(), "yyyy-MM-dd"));
    getSyncData.period.end   = QDateTime(QDate::fromString(ui->linePeriodEnd->text(),   "yyyy-MM-dd"));
    getSyncData.timeMark     = QDateTime::fromString(ui->lineTimeMark->text(), "yyyy-MM-dd hh:mm:ss.zzz");

//    QString s1 = getSyncData.period.begin.toString("yyyy-MM-dd");
//    QString s2 = getSyncData.period.end.toString("yyyy-MM-dd");
//    QString s3 = getSyncData.timeMark.toString("yyyy-MM-dd hh:mm:ss.zzz");

    lst::List<CrcItem> crcList;

    bool ok;
    getSyncData.count = ui->lineCount->text().toInt(&ok);
    if (!ok)
        getSyncData.count = 5000;

    _waitTimeout = ui->lineTimeout->text().toInt(&ok);
    if (!ok)
        _waitTimeout = _defaultWaitTimeout;
    _waitTimeout -= 2;

    simple_timer timer;
    data::GetSyncData getSyncDataAnsw;

    // Блок 1
    Message::Ptr message = createJsonMessage(getSyncData);
    message->setPriority(Message::Priority::Low);
    message->setMaxTimeLife(_waitTimeout);

    // Признак третьего примера
    message->setTag(3);

    // Отправляем запрос на первый блок данных
    if (!fomsCon().send(message))
    {
        log_error_m << u8"Соединение с ФОМС сервером отсутствует";
        return;
    }
    log_debug2_m << "Message sent"
                 << ". Id: " << message->id()
                 << ". Command: " << CommandNameLog(message->command());

    while (true)
    {
        timer.reset();
        _answerMessage.reset();

        // Ждем ответ
        while (timer.elapsed() < (_waitTimeout * 1000))
        {
            usleep(10*1000);
            QCoreApplication::processEvents();
            if (!_answerMessage.empty())
                break;
        }

        if (_answerMessage.empty())
        {
            int64_t el = timer.elapsed();
            log_error_m << "Message timer elapsed: " << el
                        << ". Wait timeout: " << _waitTimeout * 1000 << " ms";

            ui->txtGetSyncData3->append(QDateTime::currentDateTime().toString("dd.MM.yyyy hh:mm:ss.zzz"));
            ui->txtGetSyncData3->append(u8"Message timer elapsed: " + QString::number(el));
            ui->txtGetSyncData3->append(u8"Wait timeout: " + QString::number(_waitTimeout * 1000) + u8" ms");
            ui->txtGetSyncData3->append(QTEXT(u8" Превышен таймаут"));
            return;
        }

        SResult res = readFromMessage(_answerMessage, getSyncDataAnsw);
        if (!res)
        {
            log_error_m << u8"Ошибка разбора json-выражения полученного от ФОМС сервера";
            return;
        }

        QDateTime maxTimeMark = QDateTime(QDate(1900, 1, 1));

        // Определяем максимальный полученный TIME_MARK
        for (int i = 0; i < getSyncDataAnsw.items.count(); ++i)
            if (getSyncDataAnsw.items[i].TIME_MARK > maxTimeMark)
                maxTimeMark = getSyncDataAnsw.items[i].TIME_MARK;

        // Заполняем датасет для проверки CRC
        for (int i = 0; i < getSyncDataAnsw.items.count(); ++i)
            crcList.addCopy({getSyncDataAnsw.items[i].GKEY, getSyncDataAnsw.items[i].TIME_MARK});

        ui->txtGetSyncData3->append(QTEXT(u8"Получено записей: ") + QString::number(getSyncDataAnsw.items.count()));
        ui->txtGetSyncData3->append(QTEXT(u8"Максимальный TIME_MARK: ") + maxTimeMark.toString("dd.MM.yyyy hh:mm:ss.zzz"));

        if (getSyncDataAnsw.items.count() != getSyncDataAnsw.count)
        {
            // Все данные получены, выходим
            ui->txtGetSyncData3->append(QTEXT(u8"Данные получены успешно"));
            break;
        }

        //--- Подготовка данных для нового запроса ---
        // Берем максимальный timeMark от предыдущего блока данных
        getSyncData.timeMark = maxTimeMark;

//        qint64 i1 = maxTimeMark.toMSecsSinceEpoch();
//        qint64 i2 = getSyncData.timeMark.toMSecsSinceEpoch();

        // Возвращаем _waitTimeout в исходное значение
        _waitTimeout = ui->lineTimeout->text().toInt(&ok);
        if (!ok)
            _waitTimeout = _defaultWaitTimeout;
        _waitTimeout -= 2;

        message = createJsonMessage(getSyncData);
        message->setPriority(Message::Priority::Low);
        message->setMaxTimeLife(_waitTimeout);
        message->setTag(3);

        // Отправляем запрос на новый блок данных
        if (!fomsCon().send(message))
        {
            log_error_m << u8"Соединение с ФОМС сервером отсутствует";
            return;
        }
        log_debug2_m << "Message sent"
                     << ". Id: " << message->id()
                     << ". Command: " << CommandNameLog(message->command());
    } // while (true)

    bool syncDataCheck_ = true;
    config::state().getValue("sync_data.check", syncDataCheck_);

    if (!syncDataCheck_)
        return;

    crcList.sort(crcCompare);

    // Удаляем повторяющиеся записи
    for (int i = 0; i < (crcList.count() - 1); ++i)
        if (crcCompare(crcList.item(i), crcList.item(i + 1)) == 0)
            crcList.remove(i--);

    // Подсчёт контрольной суммы
    quint64 crc = 0;
    for (int i = 0; i < crcList.count(); ++i)
    {
        qint64 msecs = crcList[i].TIME_MARK.toMSecsSinceEpoch();
        quint64 value = *((quint64*)&msecs);
        crc = crc ^ value;
    }

    data::SyncDataCheck syncDataCheck;
    syncDataCheck.period.begin = QDateTime(QDate::fromString(ui->linePeriodBegin->text(), "yyyy-MM-dd"));
    syncDataCheck.period.end   = QDateTime(QDate::fromString(ui->linePeriodEnd->text(),   "yyyy-MM-dd"));
    syncDataCheck.timeMark     = QDateTime::fromString(ui->lineTimeMark->text(), "yyyy-MM-dd hh:mm:ss.zzz");

    _waitTimeout = ui->lineTimeout->text().toInt(&ok);
    if (!ok)
        _waitTimeout = _defaultWaitTimeout;
    _waitTimeout -= 2;

    data::SyncDataCheck syncDataCheckAnsw;

    message = createJsonMessage(syncDataCheck);
    message->setPriority(Message::Priority::Low);
    message->setMaxTimeLife(_waitTimeout);

    message->setTag(2);

    // Отправляем запрос на первый блок данных
    if (!fomsCon().send(message))
    {
        log_error_m << u8"Соединение с ФОМС сервером отсутствует";
        return;
    }
    log_debug2_m << "Message sent"
                 << ". Id: " << message->id()
                 << ". Command: " << CommandNameLog(message->command());

    timer.reset();
    _answerMessage.reset();

    // Ждем ответ
    while (timer.elapsed() < (_waitTimeout * 1000))
    {
        usleep(10*1000);
        QCoreApplication::processEvents();
        if (!_answerMessage.empty())
            break;
    }

    if (_answerMessage.empty())
    {
        int64_t el = timer.elapsed();
        log_error_m << "Message timer elapsed: " << el
                    << ". Wait timeout: " << _waitTimeout * 1000 << " ms";

        ui->txtGetSyncData3->append(QDateTime::currentDateTime().toString("dd.MM.yyyy hh:mm:ss.zzz"));
        ui->txtGetSyncData3->append(u8"Message timer elapsed: " + QString::number(el));
        ui->txtGetSyncData3->append(u8"Wait timeout: " + QString::number(_waitTimeout * 1000) + u8" ms");
        ui->txtGetSyncData3->append(QTEXT(u8" Превышен таймаут"));
        return;
    }

    SResult res = readFromMessage(_answerMessage, syncDataCheckAnsw);
    if (!res)
    {
        const char* msg = u8"Ошибка разбора json-выражения полученного от ФОМС сервера";
        log_error_m << msg;
        ui->txtSyncDataCheck->append(QTEXT(msg));
        return;
    }

    if (syncDataCheckAnsw.crc == crc)
    {
        ui->txtGetSyncData3->append(QTEXT(u8"CRC совпали: ") + QString::number(syncDataCheckAnsw.crc));
    }
    else
    {
        ui->txtGetSyncData3->append(QTEXT(u8"CRC не совпали") + QString::number(syncDataCheckAnsw.crc));
        ui->txtGetSyncData3->append(QTEXT(u8"Ожидалось: ") + QString::number(crc));
        ui->txtGetSyncData3->append(QTEXT(u8"Получено: ") + QString::number(syncDataCheckAnsw.crc));
    }
}

void MainWindow::on_btnSyncDataCheck_clicked(bool)
{
    data::SyncDataCheck syncDataCheck;
    syncDataCheck.period.begin = QDateTime(QDate::fromString(ui->linePeriodBeginSDC->text(), "yyyy-MM-dd"));
    syncDataCheck.period.end   = QDateTime(QDate::fromString(ui->linePeriodEndSDC->text(),   "yyyy-MM-dd"));
    syncDataCheck.timeMark     = QDateTime::fromString(ui->lineTimeMarkSDC->text(), "yyyy-MM-dd hh:mm:ss.zzz");

    bool ok;
    _waitTimeout = ui->lineTimeoutSDC->text().toInt(&ok);
    if (!ok)
        _waitTimeout = _defaultWaitTimeout;
    _waitTimeout -= 2;

    simple_timer timer;
    data::SyncDataCheck syncDataCheckAnsw;

    // Блок 1
    Message::Ptr message = createJsonMessage(syncDataCheck);
    message->setPriority(Message::Priority::Low);
    message->setMaxTimeLife(_waitTimeout);

    message->setTag(1);

    // Отправляем запрос на первый блок данных
    if (!fomsCon().send(message))
    {
        log_error_m << u8"Соединение с ФОМС сервером отсутствует";
        return;
    }
    log_debug2_m << "Message sent"
                 << ". Id: " << message->id()
                 << ". Command: " << CommandNameLog(message->command());

    timer.reset();
    _answerMessage.reset();

    // Ждем ответ
    while (timer.elapsed() < (_waitTimeout * 1000))
    {
        usleep(10*1000);
        QCoreApplication::processEvents();
        if (!_answerMessage.empty())
            break;
    }

    if (_answerMessage.empty())
    {
        int64_t el = timer.elapsed();
        log_error_m << "Message timer elapsed: " << el
                    << ". Wait timeout: " << _waitTimeout * 1000 << " ms";

        ui->txtSyncDataCheck->append(QDateTime::currentDateTime().toString("dd.MM.yyyy hh:mm:ss.zzz"));
        ui->txtSyncDataCheck->append(u8"Message timer elapsed: " + QString::number(el));
        ui->txtSyncDataCheck->append(u8"Wait timeout: " + QString::number(_waitTimeout * 1000) + u8" ms");
        ui->txtSyncDataCheck->append(QTEXT(u8" Превышен таймаут"));
        return;
    }

    SResult res = readFromMessage(_answerMessage, syncDataCheckAnsw);
    if (!res)
    {
        const char* msg = u8"Ошибка разбора json-выражения полученного от ФОМС сервера";
        log_error_m << msg;
        ui->txtSyncDataCheck->append(QTEXT(msg));
        return;
    }

    ui->txtSyncDataCheck->append(QTEXT(u8"Получен CRC: ") + QString::number(syncDataCheckAnsw.crc));
}

void MainWindow::on_btnSyncNsiVidmp_clicked(bool)
{
    Message::Ptr message = createJsonMessage(command::SyncNsiVidmp);
    message->setPriority(Message::Priority::Low);

    if (!fomsCon().send(message))
    {
        log_error_m << u8"Соединение с ФОМС сервером отсутствует";
        return;
    }

    log_debug2_m << "Message sent"
                 << ". Id: " << message->id()
                 << ". Command: " << CommandNameLog(message->command());
}

void MainWindow::command_FomsAuthorization(const Message::Ptr& message)
{
    if (message->type() == Message::Type::Answer)
    {
        if (message->execStatus() == Message::ExecStatus::Success)
        {
            fomsCon().setAuthorized(true);
            log_verbose_m << u8"Соединение c сервером ФОМС авторизовано успешно";
            _labelConnectStatus->setText("Connected (Authorized)");
        }
        else
        {
            fomsCon().setAuthorized(false);
            log_error_m << u8"Соединение c сервером ФОМС не прошло авторизацию";
            _labelConnectStatus->setText("Connected (Failed Authorization)");
        }
    }
}

void MainWindow::command_KeepWaitCommand(const Message::Ptr& message)
{
    data::KeepWaitCommand keepWaitCommand;
    readFromMessage(message, keepWaitCommand);

    if (_funcInvoker.containsCommand(keepWaitCommand.commandId))
    {
        //if (_messageId == keepWaitCommand.messageId)
        //{
        //    QMutexLocker locker(&_threadLock); (void) locker;
            log_debug2_m << "FOMS increase wait timeout: "
                         << _waitTimeout << " -> "
                         << (_waitTimeout + keepWaitCommand.timeToAdd)
                         << " for command "
                         << command::pool().commandName(keepWaitCommand.commandId)
                         << " (" << keepWaitCommand.messageId << ")";
            _waitTimeout += keepWaitCommand.timeToAdd;
        //}
    }
}

QVariant toVarchar(const QString& val)
{
    if (val != "" && val != "NA" && val != "None")
        return QVariant(val);
    else
        return QVariant(QVariant::String);
}

QVariant toVarchar(const qint32 val)
{
    return QVariant(val);
}

QVariant toSmallint(const QString& val)
{
    if (val != "" && val != "NA" && val != "None")
        return QVariant(val);
    else
        return QVariant(QVariant::Int);
}

QVariant toSmallint(const qint32 val)
{
    return QVariant((qint16)val);
}

QVariant toDate(const QString& val)
{
    if (val != "" && val != "NA" && val != "None")
        return QVariant(val);
    else
        return QVariant(QVariant::Date);
}

QVariant toDate(const qint32 val)
{
    qint64 value = qint64(val) * 1000;
    QDate date = QDateTime::fromMSecsSinceEpoch(value).date();

    if (val == 0)
        return QVariant(QVariant::Date);

    return QVariant(date);
}

QVariant toDouble(const QString& val)
{
    if (val != "" && val != "NA" && val != "None")
        return QVariant(val);
    else
        return QVariant(QVariant::Double);
}

QVariant toDouble(const double val)
{
    return QVariant(val);
}

QVariant toBigint(const QString& val)
{
    if (val != "" && val != "NA" && val != "None")
        return QVariant(val);
    else
        return QVariant(QVariant::LongLong);
}

void MainWindow::command_GetSyncData(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
    {
        log_error_m << u8"Команда GetSyncData должна иметь тип 'Answer'";
        return;
    }

    if (message->tag() == 3)
        command_GetSyncData3(message);
    else if (message->tag() == 2)
        command_GetSyncData2(message);
    else
        command_GetSyncData1(message);
}

void MainWindow::command_GetSyncData1(const Message::Ptr& message)
{
    data::GetSyncData getSyncData;
    SResult res = readFromMessage(message, getSyncData);
    if (!res)
    {
        log_error_m << u8"Ошибка разбора json-выражения полученного от ФОМС сервера";
        return;
    }

    const data::DataItem& dataItem = getSyncData.items.at(0);

    ui->txtGetSyncData1->append(QDateTime::currentDateTime().toString("dd.MM.yyyy hh:mm:ss.zzz") +
                                    " Received count = " + QString::number(getSyncData.items.count()));
    ui->txtGetSyncData1->append("");
    #define VARCHAR(COLUMN)  toVarchar(dataItem.COLUMN).toString()
    #define DOUBLE(COLUMN)   toDouble(dataItem.COLUMN).toString()
    #define SMALLINT(COLUMN) toSmallint(dataItem.COLUMN).toString()
    #define DATE(COLUMN)     toDate(dataItem.COLUMN).toString()
    #define BIGINT(COLUMN)   toBigint(dataItem.COLUMN).toString()

    ui->txtGetSyncData1->append("GKEY      = " + dataItem.GKEY.toString() );
    ui->txtGetSyncData1->append("IDSL      = " + dataItem.IDSL.toString() );
    ui->txtGetSyncData1->append("TIME_MARK = " + dataItem.TIME_MARK.toString("dd.MM.yyyy hh:mm:ss.zzz"));
    ui->txtGetSyncData1->append("OT_PER    = " + VARCHAR (OT_PER     )); // 1
    ui->txtGetSyncData1->append("MSK_OT    = " + VARCHAR (MSK_OT     )); // 2
    ui->txtGetSyncData1->append("CODE_MSK  = " + VARCHAR (CODE_MSK   )); // 3
    ui->txtGetSyncData1->append("VID_MP    = " + VARCHAR (VID_MP     )); // 4
    ui->txtGetSyncData1->append("USL_OK    = " + VARCHAR (USL_OK     )); // 5
    ui->txtGetSyncData1->append("PROFIL    = " + VARCHAR (PROFIL     )); // 6
    ui->txtGetSyncData1->append("MKB1      = " + VARCHAR (MKB1       )); // 7
    ui->txtGetSyncData1->append("MKB2      = " + VARCHAR (MKB2       )); // 8
    ui->txtGetSyncData1->append("MKB3      = " + VARCHAR (MKB3       )); // 9
    ui->txtGetSyncData1->append("CODE_USL  = " + VARCHAR (CODE_USL   )); // 10
    ui->txtGetSyncData1->append("CODE_MD   = " + VARCHAR (CODE_MD    )); // 11
    ui->txtGetSyncData1->append("KOL_USL   = " + DOUBLE  (KOL_USL    )); // 12
    ui->txtGetSyncData1->append("KOL_FACT  = " + DOUBLE  (KOL_FACT   )); // 13
    ui->txtGetSyncData1->append("ISH_MOV   = " + VARCHAR (ISH_MOV    )); // 14
    ui->txtGetSyncData1->append("RES_GOSP  = " + VARCHAR (RES_GOSP   )); // 15
    ui->txtGetSyncData1->append("TARIF_B   = " + DOUBLE  (TARIF_B    )); // 16
    ui->txtGetSyncData1->append("TARIF_S   = " + DOUBLE  (TARIF_S    )); // 17
    ui->txtGetSyncData1->append("TARIF_1K  = " + DOUBLE  (TARIF_1K   )); // 18
    ui->txtGetSyncData1->append("SUM_RUB   = " + DOUBLE  (SUM_RUB    )); // 19
    ui->txtGetSyncData1->append("VID_TR    = " + VARCHAR (VID_TR     )); // 20
    ui->txtGetSyncData1->append("EXTR      = " + SMALLINT(EXTR       )); // 21
    ui->txtGetSyncData1->append("CODE_OTD  = " + VARCHAR (CODE_OTD   )); // 22
    ui->txtGetSyncData1->append("SOUF      = " + SMALLINT(SOUF       )); // 23
    ui->txtGetSyncData1->append("SPEC_MD   = " + VARCHAR (SPEC_MD    )); // 24
    ui->txtGetSyncData1->append("DOMC_TYPE = " + VARCHAR (DOMC_TYPE  )); // 25
    ui->txtGetSyncData1->append("OKATO_INS = " + VARCHAR (OKATO_INS  )); // 26
    ui->txtGetSyncData1->append("NOVOR     = " + VARCHAR (NOVOR      )); // 27
    ui->txtGetSyncData1->append("CODE_LPU  = " + VARCHAR (CODE_LPU   )); // 28
    ui->txtGetSyncData1->append("VID_SF    = " + VARCHAR (VID_SF     )); // 29
    ui->txtGetSyncData1->append("NHISTORY  = " + VARCHAR (NHISTORY   )); // 30
    ui->txtGetSyncData1->append("PERSCODE  = " + VARCHAR (PERSCODE   )); // 31
    ui->txtGetSyncData1->append("DATE_IN   = " + DATE    (DATE_IN    )); // 32
    ui->txtGetSyncData1->append("DATE_OUT  = " + DATE    (DATE_OUT   )); // 33
    ui->txtGetSyncData1->append("TARIF_D   = " + DOUBLE  (TARIF_D    )); // 34
    ui->txtGetSyncData1->append("VID_KOEFF = " + VARCHAR (VID_KOEFF  )); // 35
    ui->txtGetSyncData1->append("USL_TMP   = " + VARCHAR (USL_TMP    )); // 35
    ui->txtGetSyncData1->append("BIRTHDAY  = " + DATE    (BIRTHDAY   )); // 36
    ui->txtGetSyncData1->append("SEX       = " + VARCHAR (SEX        )); // 37
    ui->txtGetSyncData1->append("COUNTRY   = " + VARCHAR (COUNTRY    )); // 38
    ui->txtGetSyncData1->append("SEX_P     = " + VARCHAR (SEX_P      )); // 39
    ui->txtGetSyncData1->append("BIRTHDAY_P= " + DATE    (BIRTHDAY_P )); // 40
    ui->txtGetSyncData1->append("INV       = " + SMALLINT(INV        )); // 41
    ui->txtGetSyncData1->append("DATE_NPR  = " + DATE    (DATE_NPR   )); // 42
    ui->txtGetSyncData1->append("FOR_POM   = " + SMALLINT(FOR_POM    )); // 43
    ui->txtGetSyncData1->append("MSE       = " + SMALLINT(MSE        )); // 44
    ui->txtGetSyncData1->append("P_CEL     = " + VARCHAR (P_CEL      )); // 45
    ui->txtGetSyncData1->append("DN        = " + SMALLINT(DN         )); // 46
    ui->txtGetSyncData1->append("TAL_P     = " + DATE    (TAL_P      )); // 47
    ui->txtGetSyncData1->append("PROFIL_K  = " + SMALLINT(PROFIL_K   )); // 49
    ui->txtGetSyncData1->append("NAPR_MO   = " + VARCHAR (NAPR_MO    )); // 48
    ui->txtGetSyncData1->append("MKB0      = " + VARCHAR (MKB0       )); // 49
    ui->txtGetSyncData1->append("DS_ONK    = " + SMALLINT(DS_ONK     )); // 50
    ui->txtGetSyncData1->append("VAL_KOEFF = " + DOUBLE  (VAL_KOEFF  )); // 51
    ui->txtGetSyncData1->append("C_ZAB     = " + SMALLINT(C_ZAB      )); // 52
    ui->txtGetSyncData1->append("CODE_NOM1 = " + VARCHAR (CODE_NOM1  )); // 53
    ui->txtGetSyncData1->append("CODE_NOM2 = " + VARCHAR (CODE_NOM2  )); // 54
    ui->txtGetSyncData1->append("CODE_NOM3 = " + VARCHAR (CODE_NOM3  )); // 55
    ui->txtGetSyncData1->append("VAL_TMP   = " + DOUBLE  (VAL_TMP    )); // 56
    ui->txtGetSyncData1->append("TIME_FIX  = " + VARCHAR (TIME_FIX   )); // 57
    ui->txtGetSyncData1->append("TIME_IN   = " + VARCHAR (TIME_IN    )); // 58
    ui->txtGetSyncData1->append("TIME_OUT  = " + VARCHAR (TIME_OUT   )); // 59
    ui->txtGetSyncData1->append("DATE_FIX  = " + DATE    (DATE_FIX   )); // 60
    ui->txtGetSyncData1->append("KATEG_MD  = " + VARCHAR (KATEG_MD   )); // 61
    ui->txtGetSyncData1->append("POST_MD   = " + VARCHAR (POST_MD    )); // 62
    ui->txtGetSyncData1->append("KOL_DEF   = " + DOUBLE  (KOL_DEF    )); // 63
    ui->txtGetSyncData1->append("VID_PROV  = " + VARCHAR (VID_PROV   )); // 64
    ui->txtGetSyncData1->append("MED_AREA  = " + VARCHAR (MED_AREA   )); // 65
    ui->txtGetSyncData1->append("TAL_HMP   = " + VARCHAR (TAL_HMP    )); // 66
    ui->txtGetSyncData1->append("DATE_HMP  = " + DATE    (DATE_HMP   )); // 67
    ui->txtGetSyncData1->append("MCOD_OUT  = " + VARCHAR (MCOD_OUT   )); // 68
    ui->txtGetSyncData1->append("NOM_NPR   = " + BIGINT  (NOM_NPR    )); // 69
    ui->txtGetSyncData1->append("SERIES    = " + VARCHAR (SERIES     )); // 70
    ui->txtGetSyncData1->append("NAME_MSK  = " + VARCHAR (NAME_MSK   )); // 71
    ui->txtGetSyncData1->append("OKATO_NAS = " + VARCHAR (OKATO_NAS  )); // 72
    ui->txtGetSyncData1->append("PR_LG     = " + VARCHAR (PR_LG      )); // 73

    ui->txtGetSyncData1->append("EKMP      = " + DOUBLE  (EKMP       )); // 74
    ui->txtGetSyncData1->append("MEE       = " + DOUBLE  (MEE        )); // 75

    #undef VARCHAR
    #undef DOUBLE
    #undef SMALLINT
    #undef DATE
    #undef BIGINT
}

void MainWindow::command_GetSyncData2(const Message::Ptr& message)
{
    _answerMessage = message;
}

void MainWindow::command_GetSyncData3(const Message::Ptr& message)
{
    _answerMessage = message;
}

void MainWindow::command_SyncDataCheck(const Message::Ptr& message)
{
    _answerMessage = message;
}

void MainWindow::command_SyncNsiVidmp(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
    {
        log_error_m << u8"Команда SyncNsiVidmp должна иметь тип 'Answer'";
        return;
    }

    data::SyncNsiVidmpA syncNsiVidmpA;
    SResult res = readFromMessage(message, syncNsiVidmpA);
    if (!res)
    {
        log_error_m << u8"Ошибка разбора json-выражения полученного от ФОМС сервера";
        return;
    }

    #define VAR(VAL) QVariant(VAL).toString()

    for (const data::NsiVidmp& nsiVidmp : syncNsiVidmpA.items)
    {
        ui->txtSyncNsiVidmp->append(VAR(nsiVidmp.id) + " | " + nsiVidmp.name);
    }
    #undef VAR
}

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
