#include "csv_reader.h"

#include "shared/logger/logger.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/shared/qt/config/config.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "DataReader")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "DataReader")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "DataReader")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "DataReader")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "DataReader")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "DataReader")

namespace readers {

CSVReader& readerStub()
{
    return ::safe_singleton<CSVReader, 0>();
}

CSVReader& writerStub()
{
    return ::safe_singleton<CSVReader, 1>();
}

CSVReader::CSVReader()
{

}

void CSVReader::setOptions(const QString&   filename,
                           const QDateTime& timeMark,
                           const quint64    beginTime,
                           const quint64    endTime,
                           const QChar      delimeter)
{
    _filename = filename;
    _timeMark = timeMark;
    _beginTime = beginTime;
    _endTime = endTime;
    _delimeter = delimeter;
    _pos = 0;
}

bool CSVReader::open()
{
    _pos = 0;
#ifdef MINGW
        config::base().getValue("dataset.directory", _directory);
#else
        config::base().getValue("dataset.directory", _directory);
#endif
        config::dirExpansion(_directory);

    QDirIterator it(_directory, QDir::Files, QDirIterator::NoIteratorFlags);

    while (it.hasNext())
    {
        it.next();
        //QString filePath = it.filePath();
        if(_filename == it.fileName())
        {
            _file.setFileName(it.filePath());
            break;
        }
    }
    if (_file.open(QIODevice::ReadOnly|QIODevice::Text))
    {
        log_debug_m << _filename.toStdString() << " file was opened";
        _stream.setDevice(&_file);
        return true;
    }
    else
    {
        log_error_m << _filename.toStdString() << " file cannot was opened";
        return false;
    }
}

void CSVReader::close()
{
    _file.close();
    log_debug_m << " file close";
}

QVector<QStringList> CSVReader::readRows(qint32 lines)
{
    //bool toEnd = _endTime == 0 ? true : false;

    //QList<QList<QStringList>> dataHeap;

    qint32 rows = 0;
    QTime time;
    time.start();

    QVector<QStringList> heapItem;

    if (_pos == 0)
    {
        QString line = _stream.readLine();
        _columnNames = line.split(delimeter()).toVector();
        ++_pos;
    }

    while (!_stream.atEnd())
    {
        QString line = _stream.readLine();
        if (!line.isEmpty())
        {
            QStringList list;
            list = line.split(delimeter());
            // столбец TimeMark в файле
            QString strTm = list[_columnNames.indexOf("TIME_MARK")];
            QDateTime tm = QDateTime::fromString(strTm, "dd.MM.yyyy hh:mm:ss.zzz");
            // столбец DATE_OUT в файле
            QString strDateOut = list[_columnNames.indexOf("DATE_OUT")];
            QDateTime dateOut = QDateTime::fromString(strDateOut, "yyyy-MM-dd");
            // Границы интервала
            QDateTime beg = QDateTime::fromMSecsSinceEpoch(_beginTime);
            QDateTime end = QDateTime::fromMSecsSinceEpoch(_endTime);
            // Запись должна принадлежать интервалу и иметь время модификации
            // более позднее чем полученное в команде.
            if (tm > _timeMark && ((dateOut >= beg) && (dateOut <= end)))
            {
                heapItem.append(std::move(list));
            }
            ++rows;
        }

        if (rows + 1 > lines)
            break;
    }

    return heapItem;
}

void CSVReader::writeRows(const QVector<QStringList>& data)
{
    QMutexLocker locker(&_writeLock); (void) locker;

    QVector<QStringList> dataList;

    QFile file {"/tmp/dataset_results.csv"};

    file.open(QIODevice::WriteOnly|QIODevice::Text|QIODevice::Append);

    QTextStream stream {&file};

    for (const QStringList& item : data)
    {
        for (int i = 0; i < item.size(); ++i)
        {
            stream << item[i] << ";";
        }
        stream << endl << flush;
    }
    file.close();
}

} // namespace readers
