#include "foms_emulator_appl.h"
#include "shared/defmac.h"
#include "shared/utils.h"
#include "shared/logger/logger.h"
#include "shared/logger/config.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/message.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/transport/tcp.h"
#include "shared/qt/config/config.h"
#include "shared/qt/version/version_number.h"
#include "shared/thread/thread_pool.h"

#include <QNetworkProxy>

#include <signal.h>
#include <unistd.h>

using namespace std;
using namespace communication;
using namespace communication::transport;

/**
  Используется для уведомления основного потока о завершении работы программы.
*/
void stopProgramHandler(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
    {
        const char* sigName = (sig == SIGTERM) ? "SIGTERM" : "SIGINT";
        log_verbose << "Signal " << sigName << " is received. Program will be stopped";
        Application::stop();
    }
    else
        log_verbose << "Signal " << sig << " is received";
}

void stopLog()
{
    alog::logger().flush();
    alog::logger().waitingFlush();
    alog::logger().stop();
}

void stopProgram()
{
    /*
    #define STOP_THREAD(THREAD_FUNC, NAME, TIMEOUT) \
        if (!THREAD_FUNC.stop(TIMEOUT * 1000)) { \
            log_info << "Thread '" NAME "': Timeout expired, thread will be terminated"; \
            THREAD_FUNC.terminate(); \
        }

    STOP_THREAD(videoCapture(),         "VideoCapture",       15)

    #undef STOP_THREAD
    */

    tcp::listener().close();

    log_info << "'Foms Emulator' service is stopped";
    stopLog();

    trd::threadPool().stop();
}

void helpInfo(/*const char * binary*/)
{
    alog::logger().clearSavers();
    alog::logger().addSaverStdOut(alog::Level::Info, true);

    log_info << "'Foms Emulator' service"
             << " (version: " << productVersion().toString()
             << "; protocol version: "
             << BPROTOCOL_VERSION_LOW << "-" << BPROTOCOL_VERSION_HIGH
             << "; gitrev: " << GIT_REVISION << ")";
    log_info << "Usage: foms-emulator";
    log_info << "  -n do not daemonize";
    log_info << "  -h this help";
    alog::logger().flush();
}

int main(int argc, char *argv[])
{
    // Устанавливаем в качестве разделителя целой и дробной части символ '.',
    // если этого не сделать - функции преобразования строк в числа (std::atof)
    // буду неправильно работать.
    qputenv("LC_NUMERIC", "C");

    // Пул потоков запустим после кода демонизации
    trd::threadPool().stop();

    int ret = 0;
    try
    {
        //google::InitGoogleLogging(argv[0]);
        alog::logger().start();

#ifdef NDEBUG
        alog::logger().addSaverStdOut(alog::Level::Info, true);
#else
        alog::logger().addSaverStdOut(alog::Level::Debug2);
#endif
        signal(SIGTERM, &stopProgramHandler);
        signal(SIGINT,  &stopProgramHandler);

        QDir homeDir = QDir::home();
        if (!homeDir.exists())
        {
            log_error << "Home dir " << homeDir.path() << " not exists";
            stopLog();
            return 1;
        }

        int c;
        bool isDaemon = true;
        while ((c = getopt(argc, argv, "nhc:l:")) != EOF)
        {
            switch (c)
            {
                case 'h':
                    helpInfo();
                    stopLog();
                    exit(0);
                case 'n':
                    isDaemon = false;
                    break;
                case '?':
                    log_error << "Invalid option";
                    stopLog();
                    return 1;
            }
        }

        // Путь к основному конфиг-файлу
        QString configFile = config::dir() + "/foms-emulator.conf";
        if (!QFile::exists(configFile))
        {
            log_error << "Config file " << configFile << " not exists";
            stopLog();
            return 1;
        }

        config::base().setReadOnly(true);
        config::base().setSaveDisabled(true);
        if (!config::base().readFile(configFile.toStdString()))
        {
            stopLog();
            return 1;
        }

        QString logFile;
#ifdef MINGW
        config::base().getValue("logger.file_win", logFile);
#else
        config::base().getValue("logger.file", logFile);
#endif
        config::dirExpansion(logFile);

        QFileInfo logFileInfo {logFile};
        QString logFileDir = logFileInfo.absolutePath();
        if (!QDir(logFileDir).exists())
            if (!QDir().mkpath(logFileDir))
            {
                log_error << "Failed create log directory: " << logFileDir;
                stopLog();
                return 1;
            }

        // Создаем дефолтный сэйвер для логгера
        {
            std::string logLevelStr = "info";
            config::base().getValue("logger.level", logLevelStr);

            bool logContinue = true;
            config::base().getValue("logger.continue", logContinue);

            alog::Level logLevel = alog::levelFromString(logLevelStr);
            alog::SaverPtr saver {new alog::SaverFile("default",
                                                      logFile.toStdString(),
                                                      logLevel,
                                                      logContinue)};
            alog::logger().addSaver(saver);
        }
        log_info << "'Foms Emulator' service is running"
                 << " (version " << productVersion().toString() << ")";
        alog::logger().flush();

#ifndef MINGW
        if (isDaemon)
        {
            stopLog();
            alog::logger().removeSaverStdOut();
            alog::logger().removeSaverStdErr();

            if (daemon(1, 0) != 0)
                return 0;

            alog::logger().start();
            log_verbose << "Demonization success";
        }
#endif
        alog::logger().removeSaverStdOut();
        alog::logger().removeSaverStdErr();

        // Создаем дополнительные сэйверы для логгера
        QString logConf;

#ifdef MINGW
        config::base().getValue("logger.conf_win", logConf);
#else
        config::base().getValue("logger.conf", logConf);
#endif
        config::dirExpansion(logConf);

        if (!logConf.isEmpty())
        {
            if (QFile::exists(logConf))
                alog::loadSavers(logConf.toStdString());
            else
                log_error << "Logger config file not exists: " << logConf;
        }
        alog::printSaversInfo();

        if (!communication::command::checkUnique())
        {
            stopProgram();
            return 1;
        }

        // Пул потоков нужно активировать после кода демонизации
        trd::threadPool().start();

        Application appl {argc, argv};

        // Устанавливаем текущую директорию. Эта конструкция работает только
        // когда создан экземпляр QCoreApplication.
        if (QDir::setCurrent(QCoreApplication::applicationDirPath()))
        {
            log_debug << "Set work directory: " << QCoreApplication::applicationDirPath();
        }
        else
        {
            log_error << "Failed set work directory";
            stopProgram();
            return 1;
        }

        QNetworkProxy::setApplicationProxy(QNetworkProxy::NoProxy);

        // Инициализация communication::listener::tcp
        QHostAddress hostAddress = QHostAddress::Any;
        config::readHostAddress("listener.address", hostAddress);

        int port = 62065;
        config::base().getValue("listener.port", port);
        if (!tcp::listener().init({hostAddress, port}))
        {
            stopProgram();
            return 1;
        }
        tcp::listener().setCompressionLevel(0);
        tcp::listener().setCheckProtocolCompatibility(false);

        chk_connect_q(&tcp::listener(), SIGNAL(message(communication::Message::Ptr)),
                      &appl, SLOT(message(communication::Message::Ptr)))

        chk_connect_q(&tcp::listener(), SIGNAL(socketConnected(communication::SocketDescriptor)),
                      &appl, SLOT(socketConnected(communication::SocketDescriptor)))

        chk_connect_q(&tcp::listener(), SIGNAL(socketDisconnected(communication::SocketDescriptor)),
                      &appl, SLOT(socketDisconnected(communication::SocketDescriptor)))

        if (Application::isStopped())
            return 0;

        ret = appl.exec();
    }
    catch (std::exception& e)
    {
        log_error << "Failed initialization. Detail: " << e.what();
        ret = 1;
    }
    catch (...)
    {
        log_error << "Failed initialization. Unknown error";
        ret = 1;
    }

    stopProgram();
    return ret;
}
