#include "foms_emulator_appl.h"
#include "commands/authorization.h"
#include "commands/commands.h"
#include "commands/send_score.h"
#include "commands/sync_data.h"
#include "commands/nsi.h"
#include "shared/logger/logger.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/config/config.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/logger_operators.h"
#include "shared/qt/communication/transport/tcp.h"
#include "shared/qt/communication/commands_pool.h"
#include "file_utils/csv_reader.h"

#include <QtCore>
#include <unistd.h>

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "Application")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "Application")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "Application")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "Application")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "Application")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "Application")

volatile bool Application::_stop = false;
std::atomic_int Application::_exitCode = {0};

Application::Application(int &argc, char **argv)
    : QCoreApplication(argc, argv)
{
    _stopTimerId = startTimer(1000);

    #define FUNC_REGISTRATION(COMMAND) \
        _funcInvoker.registration(command:: COMMAND, &Application::command_##COMMAND, this);

    FUNC_REGISTRATION(FomsAuthorization)
    //FUNC_REGISTRATION(NotifySendScore)
    FUNC_REGISTRATION(GetSyncData)
    FUNC_REGISTRATION(SyncDataCheck)
    FUNC_REGISTRATION(SendScore)
    //FUNC_REGISTRATION(BreakDataTransfer)
    FUNC_REGISTRATION(SyncNsiVidmp)

    #undef FUNC_REGISTRATION
}

void Application::timerEvent(QTimerEvent* event)
{
    if (event->timerId() == _stopTimerId)
    {
        if (_stop)
        {
            killTimer(_stopTimerId);
            exit(_exitCode);
            return;
        }

        // Закрываем сокеты не прошедшие авторизацию за отведенное время
        quint64 curTime = quint64(std::time(nullptr));
        if (_waitingAuthorizSocket > 0
            && _waitingAuthorizSocket < curTime)
        {
            base::Socket::List sockets = tcp::listener().sockets();
            if (sockets.empty())
            {
                _waitingAuthorizSocket = quint64(-1);
                _waitingCloseSocket = quint64(-1);
                return;
            }
            tcp::Socket::Ptr tcpSoc {dynamic_cast<tcp::Socket*>(sockets.item(0))};
            if (tcpSoc)
            {
                data::CloseConnection closeConnection;
                closeConnection.description = "Authorization timeout expired";

                Message::Ptr m = createJsonMessage(closeConnection);
                m->destinationSocketDescriptors().insert(tcpSoc->socketDescriptor());
                tcp::listener().send(m);

                log_error_m << "Authorization timeout expired"
                            << ". Remote host: " << tcpSoc->peerPoint()
                            << ". Connection will be closed";

                _waitingCloseSocket = curTime + 3;
            }
            _waitingAuthorizSocket = quint64(-1);
        }
        if (_waitingCloseSocket > 0
            && _waitingCloseSocket < curTime)
        {
            base::Socket::List sockets = tcp::listener().sockets();
            if (sockets.empty())
            {
                _waitingAuthorizSocket = quint64(-1);
                _waitingCloseSocket = quint64(-1);
                return;
            }
            tcp::Socket::Ptr tcpSoc {dynamic_cast<tcp::Socket*>(sockets.item(0))};
            if (tcpSoc)
            {
                log_error_m << "Force close of connection"
                            << ". Remote host: " << tcpSoc->peerPoint();
                tcpSoc->stop();
            }
            _waitingCloseSocket = quint64(-1);
        }
    }
}

void Application::stop(int exitCode)
{
    _exitCode = exitCode;
    stop();
}

void saveData(data::SendScore& dataMsg, quint32 taskId)
{
    using namespace readers;

    QVector<QStringList> rowsToFile;

    QStringList line = {
        dataMsg.id.toString(),
        dataMsg.name,
        dataMsg.descript,
        dataMsg.createDate.toString(),
        dataMsg.period.begin.toString(),
        dataMsg.period.end.toString(),
        dataMsg.userId.toString(),
        dataMsg.userName,
        dataMsg.modelId.toString(),
        dataMsg.modelName,
        dataMsg.modelDescr,
        dataMsg.modelPeriod.begin.toString(),
        dataMsg.modelPeriod.end.toString()
    };
    rowsToFile.push_back(std::move(line));

    writerStub().writeRows(rowsToFile);
    rowsToFile.clear();

    for (auto it = dataMsg.items.begin(); it != dataMsg.items.end(); ++it)
    {
        data::SendScoreItem sendScoreItem = (*it);
        QStringList line = {
            sendScoreItem.IDSL.toString(),
            QString::number(sendScoreItem.MEE),
            QString::number(sendScoreItem.EKMP)
        };
        rowsToFile.push_back(std::move(line));
    }
    writerStub().writeRows(rowsToFile);
}

void Application::message(const Message::Ptr& message)
{
    if (message->processed())
        return;

    if (lst::FindResult fr = _funcInvoker.findCommand(message->command()))
    {
        if (!command::pool().commandIsMultiproc(message->command()))
            message->markAsProcessed();
        _funcInvoker.call(message, fr);
    }
}

void Application::socketConnected(SocketDescriptor socketDescr)
{
    log_debug_m << "Connected to socket " << socketDescr;

    // Отказ от подключения второму клиенту
    base::Socket::List sockets = tcp::listener().sockets();
    if (sockets.count() > 1)
    {
        data::CloseConnection closeConnection;
        closeConnection.description = "Connection count exceeded";

        Message::Ptr m = createJsonMessage(closeConnection);
        m->destinationSocketDescriptors().insert(socketDescr);
        tcp::listener().send(m);
        return;
    }
    _waitingAuthorizSocket = static_cast<quint64>(std::time(nullptr) + 10);
}

void Application::socketDisconnected(SocketDescriptor socketDescr)
{
    log_info_m << "Disconnected from socket " << socketDescr;
}

void Application::command_FomsAuthorization(const Message::Ptr& message)
{
    data::FomsAuthorization fomsAuthorization;
    readFromMessage(message, fomsAuthorization);

    QString password;
    config::base().rereadFile();
    config::base().getValue("listener.password", password);

    if (password == fomsAuthorization.password)
    {
        _waitingAuthorizSocket = 0;
        _waitingCloseSocket = 0;

        // Отправляем подтверждение об успешной операции: пустое сообщение
        Message::Ptr answer = message->cloneForAnswer();
        tcp::listener().send(answer);

        log_info_m << "Success authorization from host: " << message->sourcePoint();
    }
    else
    {
        data::MessageFailed failed;
        failed.description = "Failed authorization";

        Message::Ptr answer = message->cloneForAnswer();
        writeToJsonMessage(failed, answer);
        tcp::listener().send(answer);

        data::CloseConnection closeConnection;
        closeConnection.description = failed.description;

        Message::Ptr m = createJsonMessage(closeConnection);
        m->destinationSocketDescriptors().insert(message->socketDescriptor());
        tcp::listener().send(m);

        _waitingAuthorizSocket = quint64(-1);
        _waitingCloseSocket = quint64(-1);

        log_error_m << "Failed authorization"
                    << ". Remote host: " << message->sourcePoint()
                    << ". Connection will be closed";
    }
}

void Application::command_GetSyncData(const Message::Ptr& message)
{
    if (message->tag() == 3)
        command_GetSyncData3(message);
    else if (message->tag() == 2)
        command_GetSyncData2(message);
    else
        command_GetSyncData1(message);
}

// timeMark - значение даты и времени, которое прысылает АИС Эксперт. Данное значение говорит о том,
// что необходимо выбрать только те записи, дата модификации которых равны и старше этого значения.
static QDateTime timeMark = QDateTime::fromString("01.04.2019 11:22:33.444", "dd.MM.yyyy hh:mm:ss.zzz");

static data::DataItem dataItem = {
    QUuidEx("84d1da82-4331-4972-90bc-ae5aaa68f6cd"), // -2   GKEY; QUuidEx   GKEY;
    QUuidEx(),      // -1   SLUCH_ID                               QUuidEx   IDSL;
    timeMark,       // 0    TIME_MARK;                             QDateTime TIME_MARK;
    "1805",         // 1    OT_PER;                                QString   OT_PER;
    "28",           // 2    MSK_OT;                                QString   MSK_OT;
    "28",           // 3    CODE_MSK;                              QString   CODE_MSK;
    "11",           // 4    VID_MP;                                QString   VID_MP;
    "3",            // 5    USL_OK;                                QString   USL_OK;
    "71",           // 6    PROFIL;                                QString   PROFIL ;
    "J069",         // 7    MKB1;                                  QString   MKB1;
    "",             // 8    MKB2;                                  QString   MKB2;
    "",             // 9    MKB3;                                  QString   MKB3;
    "1719600",      // 10   CODE_USL;                              QString   CODE_USL;
    "3045",         // 11   CODE_MD;                               QString   CODE_MD;
    1.0,            // 12   KOL_USL;                               double    KOL_USL = 0;
    0,              // 13   KOL_FACT;                              double    KOL_FACT = 0;
    "32",           // 14   ISH_MOV;                               QString   ISH_MOV;
    "2",            // 15   RES_GOSP;                              QString   RES_GOSP;
    276.0,          // 16   TARIF_B;                               double    TARIF_B = 0;
    276.0,          // 17   TARIF_S;                               double    TARIF_S = 0;
    0.0,            // 18   TARIF_1K;                              double    TARIF_1K = 0;
    276.0,          // 19   SUM_RUB;                               double    SUM_RUB = 0;
    "",             // 20   VID_TR;                                QString   VID_TR;
    0,              // 21   EXTR;                                  qint16    EXTR;
    "",             // 22   CODE_OTD;                              QString   CODE_OTD;
    1,              // 23   SOUF;                                  qint16    SOUF;
    "90200",        // 24   SPEC_MD;                               QString   SPEC_MD;
    "3",            // 25   DOMC_TYPE;                             QString   DOMC_TYPE;
    "46000000",     // 26   OKATO_INS;                             QString   OKATO_INS;
    "0",            // 27   NOVOR;                                 QString   NOVOR;
    "450102",       // 28   CODE_LPU;                              QString   CODE_LPU;
    "14",           // 29   VID_SF;                                QString   VID_SF;
    "2021064",      // 30   NHISTORY;                              QString   NHISTORY;
    "24352",        // 31   PERSCODE;                              QString   PERSCODE;
    1555915540,     // 32   DATE_IN;                               qint32    DATE_IN = 0;
    1555915540,     // 33   DATE_OUT;                              qint32    DATE_OUT = 0;
    0.0,            // 34   TARIF_D;                               double    TARIF_D = 0;
    "6",            // 35   VID_KOEFF;                             QString   VID_KOEFF;
    "",             // 36   USL_TMP;                               QString   USL_TMP;
    1555915540,     // 37   BIRTHDAY;                              qint32    BIRTHDAY = 0;
    "1",            // 38   SEX;                                   QString   SEX;
    "643",          // 39   COUNTRY;                               QString   COUNTRY;
    "",             // 40   SEX_P;                                 QString   SEX_P;
    1555915540,     // 41   BIRTHDAY_P;                            qint32    BIRTHDAY_P = 0;
    0,              // 42   INV;                                   qint16    INV;
    1555915540,     // 43   DATE_NPR;                              qint32    DATE_NPR = 0;
    3,              // 44   FOR_POM;                               qint16    FOR_POM = 0;
    0,              // 45   MSE;                                   qint16    MSE;
    "2.6",          // 46   P_CEL;                                 QString   P_CEL;
    0,              // 47   DN;                                    qint16    DN;
    1555915540,     // 48   TAL_P;                                 qint32    TAL_P = 0;
    0,              // 49   PROFIL_K;                              qint16    PROFIL_K;
    "",             // 50   NAPR_MO;                               QString   NAPR_MO;
    "",             // 51   MKB0;                                  QString   MKB0;
    0,              // 52   DS_ONK;                                qint16    DS_ONK = 0;
    1,              // 53   VAL_KOEFF;                             double    VAL_KOEFF = 0;
    0,              // 54   C_ZAB;                                 qint16    C_ZAB;
    "",             // 55   CODE_NOM1;                             QString   CODE_NOM1;    // TODO удалить, не используется
    "",             // 56   CODE_NOM2;                             QString   CODE_NOM2;    // TODO удалить, не используется
    "",             // 57   CODE_NOM3;                             QString   CODE_NOM3;    // TODO удалить, не используется
    "",             // 58   VAL_TMP;                               QString   VAL_TMP;      // TODO удалить, не используется
    "",             // 59   TIME_FIX;                              QString   TIME_FIX;     // TODO удалить, не используется
    "",             // 60   TIME_IN;                               QString   TIME_IN;      // TODO удалить, не используется
    "",             // 61   TIME_OUT;                              QString   TIME_OUT;     // TODO удалить, не используется
    "2001-10-07",   // 62   DATE_FIX;                              QString   DATE_FIX;     // TODO удалить, не используется
    "2",            // 63   KATEG_MD;                              QString   KATEG_MD;     // TODO удалить, не используется
    "114",          // 64   POST_MD;                               QString   POST_MD;      // TODO удалить, не используется
    "0",            // 65   KOL_DEF                                QString   KOL_DEF;      // TODO удалить, не используется
    "6",            // 66   VID_PROV                               QString   VID_PROV;     // TODO удалить, не используется
    "",             // 67   MED_AREA;                              QString   MED_AREA;     // TODO удалить, не используется
    "",             // 68   TAL_HMP;                               QString   TAL_HMP;      // TODO удалить, не используется
    1555915540,     // 69   DATE_HMP;                              qint32    DATE_HMP = 0; // TODO удалить, не используется
    "",             // 70   MCOD_OUT;                              QString   MCOD_OUT;     // TODO удалить, не используется
    "",             // 71   NOM_NPR;                               QString   NOM_NPR;      // TODO удалить, не используется
    "",             // 72   SERIES;                                QString   SERIES;       // TODO удалить, не используется
    "",             // 73   NAME_MSK;                              QString   NAME_MSK;     // TODO удалить, не используется
    "46252558000",  // 74   OKATO_NAS;                             QString   OKATO_NAS;    // TODO удалить, не используется
    "",             // 75   PR_LG;                                 QString   PR_LG;        // TODO удалить, не используется
    0,              // EKMP
    0               // MEE
};

void Application::command_GetSyncData1(const Message::Ptr& message)
{
    data::GetSyncData getSyncData;
    readFromMessage(message, getSyncData);

    Message::Ptr answer = message->cloneForAnswer();
    try
    {
        // period - период дат (границы интервала), которые прислывает АИС Эксперт.
        // Необходимово выбрать только те записи, у которых значение столбца DATE_OUT находится
        // в этом интервале. DATE_OUT >= period.begin И DATE_OUT >= period.end

        // count - запрашиваемое количество строк данных. В ответной команде должно содержаться
        // количество записей равное этому значению. Либо меньше, если данные закончились.

        // На данном этапе имеются входные данные для получения записей.
        // timeMark, period, count - используюя эти значение необходимо прочитать их из файла/БД
        // заполнить этими строками поле items в команде GetSyncData и отправить это сообщение
        // в АИС Эксперт.

        if (timeMark > getSyncData.timeMark)
        {
            // item - строка с данными, которая будет отправлена системе АИС Эксперт.
            data::DataItem item = dataItem;
            getSyncData.items.append(item);
        }

        // Заполнение сообщения для отправки данными.
        writeToJsonMessage(getSyncData, answer);

        tcp::listener().send(answer);
    }
    catch (const std::exception& e)
    {
        data::MessageFailed failed;
        failed.description = "Error on retrive input data" + QString::fromUtf8(e.what());
        writeToJsonMessage(failed, answer);
        tcp::listener().send(answer);
    }
}

void Application::command_GetSyncData2(const Message::Ptr& message)
{
    data::GetSyncData getSyncData;
    readFromMessage(message, getSyncData);

    QDateTime timeMark;
    getSyncData.items.clear();

    Message::Ptr answer = message->cloneForAnswer();

    // item - строка с данными, которая будет отправлена системе АИС Эксперт.
    data::DataItem item = dataItem;

    // Блок 2
    timeMark = QDateTime(QDate(1900, 1, 1));
    timeMark = timeMark.addMSecs(4999);

//    qint64 i1 = timeMark.toMSecsSinceEpoch();
//    qint64 i2 = getSyncData.timeMark.toMSecsSinceEpoch();

    if (getSyncData.timeMark >= timeMark)
    {
        for (int i = 0; i < 55; ++i)
        {
            item.TIME_MARK = timeMark;
            timeMark = timeMark.addMSecs(1);

            getSyncData.items.append(item);
        }

        writeToJsonMessage(getSyncData, answer);
        tcp::listener().send(answer);
        return;
    }

    // Блок 1
    timeMark = QDateTime(QDate(1900, 1, 1));
    if (getSyncData.timeMark >= timeMark)
    {
        for (int i = 0; i < 5000; ++i)
        {
            item.TIME_MARK = timeMark;
            timeMark = timeMark.addMSecs(1);

            getSyncData.items.append(item);
        }

        writeToJsonMessage(getSyncData, answer);
        tcp::listener().send(answer);
        return;
    }
}

void Application::command_GetSyncData3(const Message::Ptr& message)
{
    data::GetSyncData getSyncData;
    readFromMessage(message, getSyncData);

    QDateTime timeMark;
    getSyncData.items.clear();

    Message::Ptr answer = message->cloneForAnswer();

    // item - строка с данными, которая будет отправлена системе АИС Эксперт.
    data::DataItem item = dataItem;

    // Блок 3
    timeMark = QDateTime(QDate(1900, 1, 1));
    timeMark = timeMark.addMSecs(9998);

//    qint64 i1 = timeMark.toMSecsSinceEpoch();
//    qint64 i2 = getSyncData.timeMark.toMSecsSinceEpoch();

    if (getSyncData.timeMark >= timeMark)
    {
        data::KeepWaitCommand keepWaitCommand;
        keepWaitCommand.commandId = answer->command();
        keepWaitCommand.messageId = answer->id();
        keepWaitCommand.timeToAdd = 10;

        Message::Ptr keepMessage = createJsonMessage(keepWaitCommand);
        keepMessage->destinationSocketDescriptors().insert(answer->socketDescriptor());
        tcp::listener().send(keepMessage);

        sleep(5);

        for (int i = 0; i < 55; ++i)
        {
            item.TIME_MARK = timeMark;
            timeMark = timeMark.addMSecs(1);

            getSyncData.items.append(item);
        }

        for (int i = 0; i < 55; ++i)
            _crc.append({getSyncData.items[i].GKEY,
                         getSyncData.items[i].TIME_MARK});

        writeToJsonMessage(getSyncData, answer);
        tcp::listener().send(answer);
        return;
    }

    // Блок 2
    timeMark = QDateTime(QDate(1900, 1, 1));
    timeMark = timeMark.addMSecs(4999);

//    qint64 i1 = timeMark.toMSecsSinceEpoch();
//    qint64 i2 = getSyncData.timeMark.toMSecsSinceEpoch();

    if (getSyncData.timeMark >= timeMark)
    {
        for (int i = 0; i < getSyncData.count; ++i)
        {
            item.TIME_MARK = timeMark;
            timeMark = timeMark.addMSecs(1);

            getSyncData.items.append(item);
        }

        for (int i = 0; i < getSyncData.count; ++i)
            if (i % 2 != 0)
                getSyncData.items[i].TIME_MARK = getSyncData.items[i].TIME_MARK.addMSecs(-1);

        getSyncData.items[getSyncData.count - 1].TIME_MARK = getSyncData.items[getSyncData.count - 1].TIME_MARK.addMSecs(1);

        for (int i = 0; i < getSyncData.count; ++i)
            _crc.append({getSyncData.items[i].GKEY,
                         getSyncData.items[i].TIME_MARK});

        writeToJsonMessage(getSyncData, answer);
        tcp::listener().send(answer);
        return;
    }

    // Блок 1
    timeMark = QDateTime(QDate(1900, 1, 1));
    if (getSyncData.timeMark >= timeMark)
    {
        _crc.clear();

        for (int i = 0; i < getSyncData.count; ++i)
        {
            item.TIME_MARK = timeMark;
            timeMark = timeMark.addMSecs(1);

            getSyncData.items.append(item);
        }

        for (int i = 0; i < getSyncData.count; ++i)
            _crc.append({getSyncData.items[i].GKEY,
                         getSyncData.items[i].TIME_MARK});

        writeToJsonMessage(getSyncData, answer);
        tcp::listener().send(answer);
        return;
    }
}

void Application::command_SyncDataCheck(const Message::Ptr& message)
{
    if (message->tag() == 2)
        command_SyncDataCheck2(message);
    else
        command_SyncDataCheck1(message);
}

void Application::command_SyncDataCheck1(const Message::Ptr& message)
{
    data::SyncDataCheck syncDataCheck;
    readFromMessage(message, syncDataCheck);

    Message::Ptr answer = message->cloneForAnswer();

    // Получение данных.
    // Подсчёт контрольной суммы.
    syncDataCheck.crc = 42;

    writeToJsonMessage(syncDataCheck, answer);
    tcp::listener().send(answer);
}

void Application::command_SyncDataCheck2(const Message::Ptr& message)
{
    data::SyncDataCheck syncDataCheck;
    readFromMessage(message, syncDataCheck);

    Message::Ptr answer = message->cloneForAnswer();

    // Удаляем повторения
    for (int i = 0; i < (_crc.count() - 1); ++i)
        if (_crc[i] == _crc[i + 1])
            _crc.remove(i--);

    // Подсчёт контрольной суммы
    quint64 crc = 0;
    for (int i = 0; i < _crc.count(); ++i)
    {
        qint64 msecs = _crc[i].TIME_MARK.toMSecsSinceEpoch();
        quint64 value = *((quint64*)&msecs);
        crc = crc ^ value;
    }
    syncDataCheck.crc = crc;

    writeToJsonMessage(syncDataCheck, answer);
    tcp::listener().send(answer);
}

void Application::command_SendScore(const Message::Ptr& message)
{
    log_info_m << "Message SendScore was received from host: "
               << message->sourcePoint() << " "
               << "TaskId = " << message->tag();
    data::SendScore sendScore;
    readFromMessage(message, sendScore);
    Message::Ptr answer = message->cloneForAnswer();

    try
    {
        // Запись оценок от АИС в файл. Здесь необходима реализация
        // механизма сохранения данных для конкретной инфраструктуры.
        saveData(sendScore, message->tag());
        tcp::listener().send(answer);
    }
    catch (const std::exception& e)
    {
        data::MessageFailed failed;
        failed.description = "Error on SendScore input data" + QString::fromUtf8(e.what());
        writeToJsonMessage(failed, answer);
        tcp::listener().send(answer);
        log_error_m << "Failed SendScore for TaskId =  " << answer->tag();
    }
}

void Application::command_BreakDataTransfer(const Message::Ptr& message)
{
    log_info_m << "Message BreakDataTransfer was received from host: " << message->sourcePoint();
}

void Application::command_SyncNsiVidmp(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();

    data::SyncNsiVidmpA syncNsiVidmpA;
    data::NsiVidmp nsiVidmp;

    nsiVidmp.id = 1;
    nsiVidmp.name = QString::fromUtf8(u8"первичная медико-санитарная помощь");
    syncNsiVidmpA.items.append(nsiVidmp);

    nsiVidmp.id = 2;
    nsiVidmp.name = QString::fromUtf8(u8"скорая, в том числе специализированная, медицинская помощь");
    syncNsiVidmpA.items.append(nsiVidmp);

    nsiVidmp.id = 3;
    nsiVidmp.name = QString::fromUtf8(u8"специализированная, в том числе высокотехнологичная, медицинская помощь");
    syncNsiVidmpA.items.append(nsiVidmp);

    nsiVidmp.id = 4;
    nsiVidmp.name = QString::fromUtf8(u8"паллиативная медицинская помощь");
    syncNsiVidmpA.items.append(nsiVidmp);

    nsiVidmp.id = 11;
    nsiVidmp.name = QString::fromUtf8(u8"первичная доврачебная медико-санитарная помощь");
    syncNsiVidmpA.items.append(nsiVidmp);

    nsiVidmp.id = 12;
    nsiVidmp.name = QString::fromUtf8(u8"первичная врачебная медико-санитарная помощь");
    syncNsiVidmpA.items.append(nsiVidmp);

    nsiVidmp.id = 13;
    nsiVidmp.name = QString::fromUtf8(u8"первичная специализированная медико-санитарная помощь");
    syncNsiVidmpA.items.append(nsiVidmp);

    writeToJsonMessage(syncNsiVidmpA, answer);
    tcp::listener().send(answer);
}

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
