﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace AisFomsWorker.Dll
{
    using Newtonsoft.Json;
    using Newtonsoft.Json.Linq;
    using System.Collections.Specialized;
    using System.ComponentModel;
    using System.Xml.Serialization;

    public enum MsgType : int
	{
		Unknown = 0,
		Command = 1, // Сообщение-команда. Это может быть сообщение с командой
					 // на выполнение действия, либо это может быть запрос
					 // на получение данных. Предполагается, что в ответ
					 // на данное сообщение придет сообщение с типом Answer.
		Answer = 2, // Ответ на сообщением с типом Command.
		Event = 3  // Данный тип сообщения похож на Command, но не предполагает
				   // получения ответа (Answer). Он используется для рассылки
				   // широковещательных сообщений о событиях
	}

	// Статус выполнения/обработки сообщения. Используется в сообщениях
	// с типом Answer для того чтобы уведомить другую сторону о статусе
	// выполнения команды с типом Command
	public enum MsgExecStatus : int
	{
		Unknown = 0,
		Success = 1, // Сообщение было обработано успешно и содержит корректные
					 // ответные данные.
		Failed = 2, // Сообщение не было обработано успешно, но результат
					// не является ошибкой.
					// В данном случае сообщение (Message) будет содержать
					// данные в формате communication::data::MessageFailed.
		Error = 3  // При обработке сообщения произошла ошибка, и в качестве
				   // ответа отправляется сообщения с описанием причины ошибки.
				   // В данном случае сообщение (Message) будет содержать
				   // данные в формате communication::data::MessageError
	}

	public enum MsgPriority : int
	{
		High = 0,
		Normal = 1,
		Low = 2
		// Reserved = 3
	}

	public enum MsgCompression : int
	{
		None = 0,
		Zip = 1,
		Lzma = 2,
		Ppmd = 3,
		Disable = 7 // Используется в тех случаях когда нужно явно запретить
					// сжатие сообщения при его отправке в TCP сокет.
					// Это может потребоваться когда контент изначально сжат,
					// например, при отправке JPG, PNG, и прочих подобных
					// форматов
	}

	[Serializable]
	public class Message
	{
		private JToken _content;
		private int? _maxTimeLife;
		private BitVector32 vec = new BitVector32(0);
		private BitVector32.Section type;
		private BitVector32.Section execStatus;
		private BitVector32.Section priority;
		private BitVector32.Section compression;
		private BitVector32.Section tagsIsEmpty;
		private BitVector32.Section maxTimeLifeIsEmpty;
		private BitVector32.Section contentIsEmpty;
		private BitVector32.Section reserved2;
		private BitVector32.Section reserved3;
		private BitVector32.Section contentFormat;
		private BitVector32.Section reserved4;
		private BitVector32.Section flags2IsEmpty;

		[XmlAttribute]
		public string id { get; set; }
		
		[XmlElement]
		public string command { get; set; }

		[XmlElement]
		public uint flags
		{
			get
			{
				return (uint)vec.Data;
			}
			set
			{
				vec = new BitVector32((int)value);
				InitVector();
			}
		}

		private void InitVector()
		{
			type = BitVector32.CreateSection(7);
			execStatus = BitVector32.CreateSection(7, type);
			priority = BitVector32.CreateSection(3, execStatus);
			compression = BitVector32.CreateSection(7, priority);
			tagsIsEmpty = BitVector32.CreateSection(1, compression);
			maxTimeLifeIsEmpty = BitVector32.CreateSection(1, tagsIsEmpty);
			contentIsEmpty = BitVector32.CreateSection(1, maxTimeLifeIsEmpty);
			reserved2 = BitVector32.CreateSection(3, contentIsEmpty);
			reserved3 = BitVector32.CreateSection(255, reserved2);
			contentFormat = BitVector32.CreateSection(7, reserved3);
			reserved4 = BitVector32.CreateSection(15, contentFormat);
			flags2IsEmpty = BitVector32.CreateSection(1, reserved4);
		}

		[XmlElement]
		public int? maxTimeLife
		{
			get
			{
				return this._maxTimeLife;
			}
			set
			{
				this._maxTimeLife = value;
				vec[maxTimeLifeIsEmpty] = value.HasValue ? 0 : 1;
			}
		}

		//[JsonIgnore] public bool maxTimeLifeSpecified { get { return maxTimeLife.HasValue; } }

		[XmlIgnore]
		public JToken content
		{
			get { return _content; }
			set
			{
				this._content = value;
				vec[contentIsEmpty] = value == null ? 1 : 0;
			}
		}

		[JsonIgnore] [XmlElement]
		public MsgType MsgType
		{
			get
			{
				return (MsgType)vec[type];
			}
			set
			{
				vec[type] = (int)value;
			}
		}

		[JsonIgnore] [XmlElement]
		public MsgExecStatus MsgExecStatus
		{
			get
			{
				return (MsgExecStatus)vec[execStatus];
			}
			set
			{
				vec[execStatus] = (int)value;
			}
		}

		[JsonIgnore] [XmlElement]
		public MsgPriority MsgPriority
		{
			get
			{
				return (MsgPriority)vec[priority];
			}
			set
			{
				vec[priority] = (int)value;
			}
		}

		[JsonIgnore] [XmlElement]
		public MsgCompression MsgCompression
		{
			get
			{
				return (MsgCompression)vec[compression];
			}
			set
			{
				vec[compression] = (int)value;
			}
		}

		[JsonIgnore] [XmlElement]
		public Command MsgCommand
		{
			get
			{
				return Commands.GetCommand(this.command);
			}
			set
			{
				command = value.GetID();
			}
		}

		public Message()
		{
			InitVector();
			this.id = Guid.NewGuid().ToString();
			vec[tagsIsEmpty] = 1;
			vec[maxTimeLifeIsEmpty] = 1;
			vec[contentIsEmpty] = 1;
			vec[contentFormat] = 1;
			vec[flags2IsEmpty] = 1;
			this.MsgType = MsgType.Command;
			this.MsgExecStatus = MsgExecStatus.Unknown;
			this.MsgCompression = MsgCompression.None;
			this.MsgPriority = MsgPriority.Normal;
		}

		public Message GetAnswer(bool copyContent = false)
		{
			Message msg = new Message();
			msg.id = this.id;
			msg.command = this.command;
			msg.flags = this.flags;
			if (copyContent) {
				msg.content = this.content;
			}
			msg.MsgType = MsgType.Answer;
			msg.MsgExecStatus = MsgExecStatus.Success;
			msg.MsgCompression = MsgCompression.None;
			msg.MsgPriority = MsgPriority.Normal;
			msg.tags = this.tags;
			return msg;
		}

		[JsonIgnore] [XmlElement]
		public string Tags {
			get { return String.Join(",", (tags ?? new long[0])); }
			set { tags = value.Split(new char[] { ',' }).Select(t => Int64.Parse(t)).ToArray(); }
		}
		[JsonIgnore] public bool TagsSpecified { get { return (tags ?? new long[0]).Length > 0; } }
		[XmlIgnore] public long[] tags { get; set; }
		[JsonIgnore] public bool tagsSpecified { get { return (tags ?? new long[0]).Length > 0; } }

		public Message GetError(string error)
		{
			Message msg = GetAnswer();
			msg.id = this.id;
			msg.command = this.command;
			msg.content = JObject.FromObject(new { group = 60, code = "6027874c-7b16-4eb4-9615-ac833b54be11", description = error });
			msg.MsgExecStatus = MsgExecStatus.Error;
			return msg;
		}

		[JsonIgnore] [XmlElement("Description")] public string Descr { get; set; }
		[JsonIgnore] public bool DescrSpecified { get { return !String.IsNullOrEmpty(Descr); } }

		[XmlElement("Json_Body")] [JsonIgnore]
		public System.Xml.XmlCDataSection JsonBody
		{
			get
			{
				string jsnData = "";
				if (content == null) return new System.Xml.XmlDocument().CreateCDataSection(jsnData);
				jsnData = JsonConvert.SerializeObject(content, Formatting.Indented, 
						new JsonSerializerSettings() { NullValueHandling = NullValueHandling.Include, }
					);
				return new System.Xml.XmlDocument().CreateCDataSection(jsnData);
			}
			set {
				//string MyString = value.Value;
			}
		}

		[JsonIgnore] public bool JsonBodySpecified { get { return (this.content != null); } }

		[JsonIgnore] [XmlAttribute] public int Num { get; set; }
		[JsonIgnore] public bool NumSpecified { get { return !(Num==0); } }
	}

	/// <summary>
	/// Контент авторизации 
	/// </summary>
	public class AuthContent { [JsonProperty("password")] public string Password { get; set; } }
}
