package aisexpert.backend.spring

import aisexpert.backend.ais.AisException
import aisexpert.backend.web.HttpException
import org.slf4j.LoggerFactory
import org.springframework.http.HttpStatus
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.ControllerAdvice
import org.springframework.web.bind.annotation.ExceptionHandler
import java.net.SocketException
import java.util.concurrent.TimeoutException

data class ErrorBody(
    val group: Int,
    val code: String,
    val message: String?
)

private val criticalErrorGroups = setOf(
    10, // sql
    50, // json
    60  // ФОМС
)

@ControllerAdvice
class AisExceptionHandler {
    private val logger = LoggerFactory.getLogger(AisExceptionHandler::class.java)

    @ExceptionHandler(AisException::class)
    fun handle(ex: AisException): ResponseEntity<ErrorBody> {
        val status = if (ex.group in criticalErrorGroups) {
            logger.error("Unknown AIS error", ex)
            HttpStatus.INTERNAL_SERVER_ERROR
        } else {
            HttpStatus.BAD_REQUEST
        }
        return httpError(status, ex.group, ex.code, ex.message)
    }

    @ExceptionHandler(HttpException::class)
    fun handle(ex: HttpException): ResponseEntity<ErrorBody> {
        return httpError(ex.status, ex.group, ex.code, ex.message)
    }

    @ExceptionHandler(SocketException::class)
    fun handle(ex: SocketException): ResponseEntity<ErrorBody> {
        return httpError(HttpStatus.SERVICE_UNAVAILABLE)
    }

    @ExceptionHandler(TimeoutException::class)
    fun handle(ex: TimeoutException): ResponseEntity<ErrorBody> {
        return httpError(HttpStatus.REQUEST_TIMEOUT)
    }

    fun handle(ex: Throwable) = when (ex) {
        is AisException -> handle(ex)
        is HttpException -> handle(ex)
        is SocketException -> handle(ex)
        is TimeoutException -> handle(ex)
        else -> httpError(HttpStatus.INTERNAL_SERVER_ERROR)
    }

    private fun httpError(
        status: HttpStatus,
        group: Int = 0,
        code: String = "",
        message: String? = null
    ): ResponseEntity<ErrorBody> {
        return ResponseEntity(ErrorBody(group, code, message ?: status.reasonPhrase), status)
    }
}