package aisexpert.backend.spring

import aisexpert.backend.ais.AisConnection
import aisexpert.backend.ais.Message
import aisexpert.backend.ais.MessageType
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.config.BeanPostProcessor
import org.springframework.core.Ordered
import org.springframework.stereotype.Component
import java.lang.reflect.Method

@Component
class AisListenerBeanPostProcessor(
    private val connection: AisConnection
) : Ordered, BeanPostProcessor {
    private val logger = LoggerFactory.getLogger(AisListenerBeanPostProcessor::class.java)
    private val beans = HashMap<String, Class<*>>()

    override fun getOrder() = Ordered.LOWEST_PRECEDENCE

    override fun postProcessBeforeInitialization(bean: Any, beanName: String): Any? {
        val clazz = bean.javaClass
        if (clazz.isAnnotationPresent(AisListener::class.java)) {
            beans[beanName] = clazz
        }

        return bean
    }

    override fun postProcessAfterInitialization(bean: Any, beanName: String): Any? {
        val clazz = beans[beanName] ?: return bean
        val commandMethods = clazz.methods
            .filter { it.isAnnotationPresent(AisCommandHandler::class.java) }
        val eventMethods = clazz.methods
            .filter { it.isAnnotationPresent(AisEventHandler::class.java) }

        logger.info("Found AIS listener [${clazz.name}]")

        for (method in commandMethods) {
            val annotation = method.getAnnotation(AisCommandHandler::class.java)
            registerMethod(bean, method, annotation.type, MessageType.COMMAND)
            logger.info("Command handler registered [$method]")
        }

        for (method in eventMethods) {
            val annotation = method.getAnnotation(AisEventHandler::class.java)
            registerMethod(bean, method, annotation.type, MessageType.EVENT)
            logger.info("Event handler registered [$method]")
        }

        return bean
    }

    private fun registerMethod(bean: Any, method: Method, commandType: String, messageType: MessageType) {
        val tagIndex = aisTagIndex(method)
        val commandIndex = aisCommandIndex(method)

        if (commandIndex != null) {
            connection.registerContentType(messageType, commandType, method.parameterTypes[commandIndex])
        }
        if (messageType == MessageType.COMMAND && method.returnType != Void::class.javaPrimitiveType) {
            connection.registerContentType(MessageType.ANSWER, commandType, method.returnType)
        }

        connection.registerCommandHandler(commandType) { message ->
            val args = Array<Any?>(method.parameterCount) { null }
            tagIndex?.let { args[tagIndex] = message.tags?.singleOrNull() }
            commandIndex?.let { args[commandIndex] = message.content }
            val result = method.invoke(bean, *args)
            if (messageType == MessageType.COMMAND) {
                connection.sendMessage(Message(
                    id = message.id,
                    type = MessageType.ANSWER,
                    command = message.command,
                    content = if (result == Unit) null else result
                ))
            }
        }
    }
}