package aisexpert.backend.spring

import org.springframework.beans.factory.config.ConfigurableListableBeanFactory
import org.springframework.beans.factory.support.BeanDefinitionBuilder
import org.springframework.beans.factory.support.BeanDefinitionRegistry
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor
import org.springframework.core.type.filter.AnnotationTypeFilter
import org.springframework.stereotype.Component

/**
 * Класс для регистрации интерфейсов помеченных аннотацией [AisService].
 * Все методы интерфейса должны быть помечены аннотацией [AisCommand] для указания UUID типа команды.
 * Метод может иметь один аргумент - команду или ноль аргументов - команда пустая.
 * Возвращаемое значение метода - ответ от сервиса.
 */
@Component
class AisServiceBeanFactoryPostProcessor : BeanDefinitionRegistryPostProcessor {
    override fun postProcessBeanDefinitionRegistry(registry: BeanDefinitionRegistry) {
        val scanner = ClassPathScanningCandidateInterfaceProvider()
        scanner.addIncludeFilter(AnnotationTypeFilter(AisService::class.java))
        val serviceDefinitions = scanner.findCandidateComponents("aisexpert.backend")

        for (serviceDefinition in serviceDefinitions) {
            val iface = Class.forName(serviceDefinition.beanClassName)
            val beanDefinition = BeanDefinitionBuilder.rootBeanDefinition(iface)
                .setFactoryMethodOnBean("create", "aisServiceFactory")
                .addConstructorArgValue(iface)
                .beanDefinition
            registry.registerBeanDefinition(iface.simpleName, beanDefinition)
        }
    }

    override fun postProcessBeanFactory(beanFactory: ConfigurableListableBeanFactory) {}
}