package aisexpert.backend.spring

import aisexpert.backend.ais.AisConnection
import aisexpert.backend.ais.Message
import aisexpert.backend.ais.MessageType
import org.springframework.stereotype.Component
import java.lang.reflect.Method
import java.lang.reflect.Proxy

/**
 * Фабрика, которая создаёт реализации для интерфейсов помеченных аннотацией [AisService].
 */
@Component("aisServiceFactory")
class AisServiceFactory(
    private val connection: AisConnection
) {
    @Suppress("UNCHECKED_CAST", "unused")
    fun <T> create(iface: Class<T>): T {
        val handlers = iface.methods
            .associate { Pair(it, createHandler(it)) }
        val obj = Any()

        return Proxy.newProxyInstance(this.javaClass.classLoader, arrayOf(iface)) { _, method, args ->
            val handler = handlers[method]
            if (handler != null) {
                handler(args)
            } else {
                // перенаправление методов класса Object
                // спринг вызывает hashCode/equals чтобы формировать мапу синглтонов
                method.invoke(obj, *(args ?: emptyArray()))
            }
        } as T
    }

    private fun createHandler(method: Method): (Array<Any?>?) -> Any? {
        val tagIndex = aisTagIndex(method)
        val commandIndex = aisCommandIndex(method)

        var answerJavaType = method.returnType
        answerJavaType = if (answerJavaType == Void::class.javaPrimitiveType) Any::class.java else answerJavaType

        val commandType = method.getAnnotation(AisCommand::class.java)?.type
            ?: throw RuntimeException("Method [$method] must be annotated with @AisCommand")

        if (commandIndex != null) {
            connection.registerContentType(MessageType.COMMAND, commandType, method.parameterTypes[commandIndex])
        }
        connection.registerContentType(MessageType.ANSWER, commandType, answerJavaType)

        return { args ->
            val content = commandIndex?.let { args?.get(it) }
            val tag = tagIndex?.let { args?.get(it) as Long }

            val id = connection.sendMessage(Message(
                type = MessageType.COMMAND,
                command = commandType,
                content = content,
                tags = tag?.let { listOf(it) } ?: emptyList()
            ))
            val answerMessage = connection.waitForAnswer(commandType, id)
            answerMessage.content
        }
    }
}