package aisexpert.backend.spring

import aisexpert.backend.entity.AuthToken
import com.fasterxml.classmate.types.ResolvedObjectType
import com.google.common.base.Predicate
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import springfox.documentation.builders.ApiInfoBuilder
import springfox.documentation.builders.PathSelectors
import springfox.documentation.builders.RequestHandlerSelectors
import springfox.documentation.schema.AlternateTypeRule
import springfox.documentation.service.ApiKey
import springfox.documentation.service.AuthorizationScope
import springfox.documentation.service.SecurityReference
import springfox.documentation.spi.DocumentationType
import springfox.documentation.spi.service.contexts.SecurityContext
import springfox.documentation.spring.web.plugins.Docket
import springfox.documentation.swagger.web.SecurityConfiguration
import springfox.documentation.swagger.web.SecurityConfigurationBuilder
import springfox.documentation.swagger2.annotations.EnableSwagger2

class AuthTokenDocs

@Configuration
@EnableSwagger2
class SwaggerConfiguration {
    @Bean
    fun api(): Docket {
        return Docket(DocumentationType.SWAGGER_2)
            .securityContexts(listOf(securityContext()))
            .securitySchemes(listOf(securityScheme()))
            .apiInfo(ApiInfoBuilder()
                .title("Ais Expert Backend Proxy")
                .description("Ais Expert Backend Proxy")
                .version(Package.getPackage("aisexpert.backend").implementationVersion)
                .build())
            .alternateTypeRules(
                AlternateTypeRule(
                    ResolvedObjectType.create(AuthToken::class.java, null, null, null),
                    ResolvedObjectType.create(AuthTokenDocs::class.java, null, null, null)
                )
            )
            .select()
            .apis(RequestHandlerSelectors.basePackage("aisexpert.backend"))
            .build()
    }

    @Bean
    fun securityConfiguration(): SecurityConfiguration = SecurityConfigurationBuilder.builder().build()

    private fun securityScheme() = ApiKey("Token", TOKEN_HEADER_KEY, "header")

    private fun securityContext() = SecurityContext.builder()
        .securityReferences(defaultAuth())
        .forPaths(combineSelectors(
            PathSelectors.ant("/api/**"),
            PathSelectors.ant("/api/auth/login").not()
        ))
        .build()

    private fun defaultAuth() = listOf(SecurityReference("Token", arrayOf(
        AuthorizationScope("global", "accessEverything")
    )))

    private fun combineSelectors(vararg selectors: Predicate<String>): Predicate<String> {
        return Predicate { str -> selectors.all { it.apply(str) } }
    }

    private fun Predicate<String>.not() = Predicate<String> { !this.apply(it) }
}