package aisexpert.backend.spring

import aisexpert.backend.repository.AuthTokenRepository
import com.fasterxml.jackson.databind.ObjectMapper
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.web.filter.OncePerRequestFilter
import javax.servlet.FilterChain
import javax.servlet.http.HttpServletRequest
import javax.servlet.http.HttpServletResponse

const val TOKEN_PREFIX = "Bearer "
const val TOKEN_HEADER_KEY = "Authorization"
const val TOKEN_QUERY_KEY = "auth-token"

class WebSecurityTokenFilter(
	private val authTokenRepository: AuthTokenRepository,
	private val objectMapper: ObjectMapper
) : OncePerRequestFilter() {
	override fun doFilterInternal(request: HttpServletRequest, response: HttpServletResponse, filterChain: FilterChain) {
		var header = request.getHeader(TOKEN_HEADER_KEY)
		header = header ?: request.getParameter(TOKEN_QUERY_KEY)

		if (header == null) {
			filterChain.doFilter(request, response)
			return
		}

		if (!header.startsWith(TOKEN_PREFIX)) {
			response.status = HttpServletResponse.SC_BAD_REQUEST
			response.writeJson(objectMapper, ErrorBody(
				group = 0,
				code = "",
				message = "Authorization token format must be 'Bearer <token>'"
			))
			return
		}

		val secret = header.replace(TOKEN_PREFIX, "")
		val token = authTokenRepository.findById(secret).orElse(null)

		if (token == null) {
			response.status = HttpServletResponse.SC_FORBIDDEN
			response.writeJson(objectMapper, ErrorBody(
				group = 0,
				code = "",
				message = "Authorization token not found"
			))
			return
		}

		SecurityContextHolder.getContext().authentication = token
		filterChain.doFilter(request, response)
	}
}