package aisexpert.backend.web

import aisexpert.backend.ais.AisUserService
import aisexpert.backend.ais.UserAuthCommand
import aisexpert.backend.entity.AuthToken
import aisexpert.backend.repository.AuthTokenRepository
import aisexpert.backend.spring.AisCommandHandler
import aisexpert.backend.spring.AisListener
import aisexpert.backend.spring.AisTag
import aisexpert.backend.spring.TOKEN_PREFIX
import org.springframework.transaction.annotation.Transactional
import org.springframework.web.bind.annotation.*
import java.time.OffsetDateTime
import java.util.*
import javax.servlet.http.HttpServletRequest

data class UserLoginRequest(
    val login: String,
    val password: String,
    val removeOldTokens: Boolean
)

data class UserLoginResponse(
    val token: String
)

data class AuthTokenDto(
    val secret: String,
    val userAgent: String?,
    val clientIp: String?,
    val createdAt: OffsetDateTime?,
    val lastActive: OffsetDateTime?
)

data class RevokeAuthTokenDto(
    val secrets: List<String>
)

@RestController
@RequestMapping("/api/auth")
@AisListener
class AuthController(
    private val aisUserService: AisUserService,
    private val authTokenRepository: AuthTokenRepository,
    private val webSocketRegistry: WebSocketRegistry
) {
    @PostMapping("/login")
    @Transactional
    fun login(
        @RequestBody data: UserLoginRequest,
        request: HttpServletRequest
    ): UserLoginResponse {
        val authAnswer = aisUserService.userAuth(UserAuthCommand(data.login, data.password))

        if (data.removeOldTokens) {
            val tokens = authTokenRepository.findByHashId(authAnswer.hashId)
            tokens.forEach { webSocketRegistry.disconnectByPrincipal(it) }
            authTokenRepository.deleteAll(tokens)
        }

        val now = OffsetDateTime.now()

        val token = AuthToken(
            secret = UUID.randomUUID().toString(),
            userId = authAnswer.id,
            userLogin = data.login,
            hashId = authAnswer.hashId,
            isAdmin = authAnswer.isAdmin,
            userAgent = request.getHeader("User-Agent"),
            clientIp = request.remoteAddr,
            createdAt = now,
            lastActive = now
        )
        authTokenRepository.save(token)

        return UserLoginResponse(
            token = TOKEN_PREFIX + token.secret
        )
    }

    @PostMapping("/logout")
    fun logout(authToken: AuthToken) {
        aisUserService.userLogout(authToken.hashId)
        authTokenRepository.delete(authToken)
        webSocketRegistry.disconnectByPrincipal(authToken)
    }

    @GetMapping("/tokens")
    fun tokens(authToken: AuthToken): List<AuthTokenDto> {
        return authTokenRepository.findByHashId(authToken.hashId).map {
            AuthTokenDto(
                secret = it.secret,
                userAgent = it.userAgent,
                clientIp = it.clientIp,
                createdAt = it.createdAt,
                lastActive = it.lastActive
            )
        }
    }

    @PostMapping("/tokens/revoke")
    @Transactional
    fun revoke(authToken: AuthToken, @RequestBody data: RevokeAuthTokenDto) {
        val tokens = authTokenRepository.findAllById(data.secrets)
        tokens.forEach(this::logout)
    }

    @AisCommandHandler(type = "14c292a9-5e92-4c8d-8f61-ba3658254754")
    @Transactional
    fun resetUserAuth(@AisTag hashId: Long) {
        for (token in authTokenRepository.findByHashId(hashId)) {
            webSocketRegistry.disconnectByPrincipal(token)
        }
        authTokenRepository.deleteByHashId(hashId)
    }
}