package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.entity.AuthToken
import org.springframework.web.bind.annotation.*
import java.time.OffsetDateTime

data class DataSyncWrapperDto(
    val taskId: String,
    val isEnabled: Boolean,
    val runDateTime: OffsetDateTime,
    val nextDateTime: OffsetDateTime,
    val execStatus: TaskExecStatus,
    val attemptLimit: Int,
    val attemptInterval: Int,
    val periodDuration: Short,
    val current: Int,
    val total: Int
)

data class DataSyncUpdateDto(
    val isEnabled: Boolean? = null,
    val nextDateTime: OffsetDateTime? = null,
    val attemptLimit: Int? = null,
    val attemptInterval: Int? = null,
    val periodDuration: Short? = null
)

@RestController
@RequestMapping("/api/data-sync")
class DataSyncController(
    private val aisSyncDataService: AisSyncDataService,
    private val aisTaskService: AisTaskService
) {
    @GetMapping
    fun info(authToken: AuthToken): DataSyncWrapperDto {
        val inner = aisSyncDataService.info(authToken.hashId)
        return inner.toWrapper()
    }

    @PatchMapping
    fun update(authToken: AuthToken, @RequestBody data: DataSyncUpdateDto): DataSyncWrapperDto {
        var info = aisSyncDataService.info(authToken.hashId)
        info = info.copy(
            isEnabled = data.isEnabled ?: info.isEnabled,
            nextDateTime = data.nextDateTime ?: info.nextDateTime,
            attemptInterval = data.attemptInterval ?: info.attemptInterval,
            attemptLimit = data.attemptLimit ?: info.attemptLimit,
            relPeriodBegin = 0,
            relPeriodDuration = data.periodDuration ?: info.relPeriodDuration
        )
        aisSyncDataService.edit(authToken.hashId, SyncDataEditCommand(
            isEnabled = info.isEnabled,
            runDateTime = info.runDateTime,
            nextDateTime = info.nextDateTime,
            attemptInterval = info.attemptInterval,
            attemptLimit = info.attemptLimit,
            relPeriodBegin = info.relPeriodBegin,
            relPeriodDuration = info.relPeriodDuration
        ))
        return info.toWrapper()
    }

    @PostMapping("/now")
    fun syncNow(authToken: AuthToken) {
        aisSyncDataService.now(authToken.hashId)
    }

    @PostMapping("/interrupt")
    fun interrupt(authToken: AuthToken) {
        val task = aisSyncDataService.info(authToken.hashId)
        aisTaskService.interrupt(authToken.hashId, TaskInterruptCommand(
            taskId = task.taskId,
            taskType = TaskType.SYNC_DATA,
            userId = authToken.userId
        ))
    }

    private fun SyncDataDto.toWrapper() = DataSyncWrapperDto(
        taskId = taskId,
        isEnabled = isEnabled,
        attemptInterval = attemptInterval,
        attemptLimit = attemptLimit,
        nextDateTime = nextDateTime,
        periodDuration = relPeriodDuration,
        runDateTime = runDateTime,
        execStatus = taskExecStatus,
        current = current,
        total = total
    )
}