package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.entity.AuthToken
import org.springframework.web.bind.annotation.*

data class GroupZonesDto(
    val learn: ZoneRight,
    val apply: ZoneRight,
    val reports: ZoneRight,
    val supervising: ZoneRight,
    val reportsFed: ZoneRight
)

data class GroupDto(
    val id: String,
    val name: String,
    val zones: GroupZonesDto,
    val users: Int,
    val isDefault: Boolean
)

data class GroupZonesUpdateDto(
    val learn: ZoneRight? = null,
    val apply: ZoneRight? = null,
    val reports: ZoneRight? = null,
    val supervising: ZoneRight? = null,
    val reportsFed: ZoneRight? = null
)

data class GroupUpdateDto(
    override val id: String,
    val name: String? = null,
    val isDefault: Boolean? = null,
    val zones: GroupZonesUpdateDto? = null
) : BatchItem

data class GroupCreateDto(
    val name: String,
    val zones: GroupZonesDto
)

@RestController
@RequestMapping("/api/groups")
class GroupController(
    private val aisRoleService: AisRoleService,
    private val batchUpdateService: BatchUpdateService
) {
    @GetMapping
    fun list(authToken: AuthToken): List<GroupDto> {
        return aisRoleService.list(authToken.hashId, RoleListCommand(listOf())).items.map(AisRole::toDto)
    }

    @PostMapping
    fun create(authToken: AuthToken, @RequestBody data: GroupCreateDto): GroupDto {
        var role = AisRole(
            id = "",
            name = data.name,
            canApplyModel = data.zones.apply,
            canLearnModel = data.zones.learn,
            canMonitoring = data.zones.supervising,
            canRWReport = data.zones.reports,
            canReportFed = data.zones.reportsFed,
            isDefault = false,
            userCount = 0
        )
        role = aisRoleService.add(authToken.hashId, role)
        return role.toDto()
    }

    @PatchMapping("/update-multiple")
    fun update(authToken: AuthToken, @RequestBody data: BatchUpdate<GroupUpdateDto>): BatchResult<GroupDto> {
        return batchUpdateService(data, { group ->
            var role = aisRoleService.info(authToken.hashId, RoleInfoCommand(group.id))
            role = AisRole(
                id = group.id,
                name = group.name ?: role.name,
                canLearnModel = group.zones?.learn ?: role.canLearnModel,
                canApplyModel = group.zones?.apply ?: role.canApplyModel,
                canMonitoring = group.zones?.supervising ?: role.canMonitoring,
                canRWReport = group.zones?.reports ?: role.canRWReport,
                canReportFed = group.zones?.reportsFed ?: role.canReportFed,
                isDefault = group.isDefault ?: role.isDefault,
                userCount = role.userCount
            )
            aisRoleService.update(authToken.hashId, role)
            role.toDto()
        }, { id ->
            aisRoleService.delete(authToken.hashId, DeleteRoleCommand(id))
        })
    }

    @GetMapping("/{groupId}")
    fun get(authToken: AuthToken, @PathVariable groupId: String): GroupDto {
        return aisRoleService.info(authToken.hashId, RoleInfoCommand(groupId)).toDto()
    }
}

fun AisRole.toDto() = GroupDto(
    id = id,
    name = name,
    zones = GroupZonesDto(
        learn = canLearnModel,
        apply = canApplyModel,
        reports = canRWReport,
        supervising = canMonitoring,
        reportsFed = canReportFed
    ),
    isDefault = isDefault,
    users = userCount
)