package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.entity.AuthToken
import org.springframework.web.bind.annotation.*
import java.time.OffsetDateTime

data class ModelListDto(
    val total: Int,
    val items: List<ModelDto>
)

data class ModelUpdateDto(
    override val id: String,
    val name: String?,
    val description: String?,
    val isPublic: Boolean?
) : BatchItem

@RestController
@RequestMapping("/api/models")
class ModelController(
    private val aisModelService: AisModelService,
    private val batchUpdateService: BatchUpdateService
) {
    @GetMapping
    fun list(
        authToken: AuthToken,
        @RequestParam(required = false) modelName: String? = null,
        @RequestParam(required = false) userId: String? = null,
        @RequestParam(required = false) learnDateFrom: OffsetDateTime? = null,
        @RequestParam(required = false) learnDateTo: OffsetDateTime? = null,
        @RequestParam(required = false) createDateFrom: OffsetDateTime? = null,
        @RequestParam(required = false) createDateTo: OffsetDateTime? = null,
        @RequestParam(required = false) show: ShowFilter? = null,
        @RequestParam(required = false) otherUsers: List<String>? = null,
        @RequestParam(required = false) taskModelId: String? = null,
        @RequestParam(required = false) archive: ArchiveFilter? = null,
        @RequestParam(required = false) offset: Int? = null,
        @RequestParam(required = false) limit: Int? = null
    ): ModelListDto {
        val models = aisModelService.list(authToken.hashId, ModelListCommand(
            filter = ModelFilter(
                userId = userId ?: authToken.userId,
                name = modelName ?: "",
                createDate = TimeRange(
                    begin = createDateFrom,
                    end = createDateTo
                ),
                dataPeriod = TimeRange(
                    begin = learnDateFrom,
                    end = learnDateTo
                ),
                show = show ?: ShowFilter.SELF,
                otherUsers = otherUsers ?: emptyList(),
                taskId = taskModelId,
                archive = archive ?: ArchiveFilter.NO
            )
        )).items
        return ModelListDto(
            total = 100,
            items = models
        )
    }

    @GetMapping("/{modelId}")
    fun info(authToken: AuthToken, @PathVariable modelId: String): ModelDto {
        return aisModelService.info(authToken.hashId, ModelInfoCommand(modelId, authToken.userId))
    }

    @PatchMapping("/update-multiple")
    fun update(authToken: AuthToken, @RequestBody data: BatchUpdate<ModelUpdateDto>): BatchResult<ModelDto> {
        return batchUpdateService(data, { update ->
            val model = aisModelService.info(authToken.hashId, ModelInfoCommand(update.id, authToken.userId))
            val edit = ModelEditCommand(
                id = model.id,
                userId = model.userId,
                name = update.name ?: model.name,
                description = update.description ?: model.description,
                isPublic = update.isPublic ?: model.isPublic
            )
            aisModelService.edit(authToken.hashId, edit)
            model.copy(
                name = edit.name,
                description = edit.description,
                isPublic = edit.isPublic
            )
        }, { id ->
            aisModelService.delete(authToken.hashId, ModelDeleteCommand(id, authToken.userId, data.forceDelete))
        })
    }
}