package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.entity.AuthToken
import io.swagger.annotations.ApiModelProperty
import org.springframework.web.bind.annotation.*

data class NotifyAddressCreateDto(
    val name: String,
    val email: String,
    val triggers: List<String>
)

data class NotifyAddressUpdateDto(
    override val id: String,
    val name: String? = null,
    val email: String? = null,
    val triggers: List<String>? = null
) : BatchItem

data class NotifyTriggerCreateDto(
    @ApiModelProperty("короткое имя триггера (30 символов)")
    val name: String,

    @ApiModelProperty("тип триггера: cpu (%), ram (%), hdd (%), cputmp (*C), foms (да/нет)")
    val type: String,

    @ApiModelProperty("порог срабатывания")
    val threshold: Int,

    @ApiModelProperty("интервал отправки повторного сообщения (в минутах)")
    val emissionInterval: Int,

    @ApiModelProperty("тема почтового сообщения (100 символов)")
    val subject: String,

    @ApiModelProperty("тело почтового сообщения (1000 символов)")
    val messageTemplate: String
)

data class NotifyTriggerUpdateDto(
    override val id: String,

    @ApiModelProperty("короткое имя триггера (30 символов)")
    val name: String? = null,

    @ApiModelProperty("тип триггера: cpu (%), ram (%), hdd (%), cputmp (*C), foms (да/нет)")
    val type: String? = null,

    @ApiModelProperty("порог срабатывания")
    val threshold: Int? = null,

    @ApiModelProperty("интервал отправки повторного сообщения (в минутах)")
    val emissionInterval: Int? = null,

    @ApiModelProperty("тема почтового сообщения (100 символов)")
    val subject: String? = null,

    @ApiModelProperty("тело почтового сообщения (1000 символов)")
    val messageTemplate: String? = null
) : BatchItem

@RestController
@RequestMapping("/api/notify")
class NotifyController(
    private val notifyAddressService: AisNotifyAddressService,
    private val notifyTriggerService: AisNotifyTriggerService,
    private val batchUpdateService: BatchUpdateService
) {
    @GetMapping("/addresses")
    fun listAddresses(authToken: AuthToken): List<NotifyAddressDto> {
        return notifyAddressService.list(authToken.hashId, NotifyAddressListCommand()).items
    }

    @PostMapping("/addresses")
    fun createAddress(authToken: AuthToken, @RequestBody data: NotifyAddressCreateDto): NotifyAddressDto {
        return notifyAddressService.create(authToken.hashId, NotifyAddressCreateCommand(
            name = data.name,
            email = data.email,
            triggers = data.triggers
        ))
    }

    @GetMapping("/addresses/{id}")
    fun getAddress(authToken: AuthToken, @PathVariable id: String): NotifyAddressDto {
        return notifyAddressService.info(authToken.hashId, NotifyAddressInfoCommand(id))
    }

    @PatchMapping("/addresses/update-multiple")
    fun updateAddresses(authToken: AuthToken, @RequestBody data: BatchUpdate<NotifyAddressUpdateDto>): BatchResult<NotifyAddressDto> {
        return batchUpdateService(data, { update ->
            var address = notifyAddressService.info(authToken.hashId, NotifyAddressInfoCommand(update.id))
            address = address.copy(
                name = update.name ?: address.name,
                email = update.email ?: address.email,
                triggers = update.triggers ?: address.triggers
            )
            notifyAddressService.edit(authToken.hashId, address)
            address
        }, { id ->
            notifyAddressService.delete(authToken.hashId, NotifyAddressDeleteCommand(id))
        })
    }

    @GetMapping("/triggers")
    fun listTriggers(authToken: AuthToken): List<NotifyTriggerDto> {
        return notifyTriggerService.list(authToken.hashId, NotifyTriggerListCommand()).items
    }

    @PostMapping("/triggers")
    fun createTrigger(authToken: AuthToken, @RequestBody data: NotifyTriggerCreateDto): NotifyTriggerDto {
        return notifyTriggerService.create(authToken.hashId, NotifyTriggerCreateCommand(
            name = data.name,
            type = data.type,
            emissionInterval = data.emissionInterval,
            messageTemplate = data.messageTemplate,
            subject = data.subject,
            threshold = data.threshold
        ))
    }

    @GetMapping("/triggers/{id}")
    fun getTrigger(authToken: AuthToken, @PathVariable id: String): NotifyTriggerDto {
        return notifyTriggerService.info(authToken.hashId, NotifyTriggerInfoCommand(id))
    }

    @PatchMapping("/triggers/update-multiple")
    fun updateTriggers(authToken: AuthToken, @RequestBody data: BatchUpdate<NotifyTriggerUpdateDto>): BatchResult<NotifyTriggerDto> {
        return batchUpdateService(data, { update ->
            var trigger = notifyTriggerService.info(authToken.hashId, NotifyTriggerInfoCommand(update.id))
            trigger = trigger.copy(
                name = update.name ?: trigger.name,
                threshold = update.threshold ?: trigger.threshold,
                subject = update.subject ?: trigger.subject,
                messageTemplate = update.messageTemplate ?: trigger.messageTemplate,
                emissionInterval = update.emissionInterval ?: trigger.emissionInterval,
                type = update.type ?: trigger.type
            )
            notifyTriggerService.edit(authToken.hashId, trigger)
            trigger
        }, { id ->
            notifyTriggerService.delete(authToken.hashId, NotifyTriggerDeleteCommand(id))
        })
    }
}