package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.entity.AuthToken
import org.springframework.web.bind.annotation.*
import java.time.OffsetDateTime

data class ReportListDto(
    val total: Int,
    val items: List<ReportDto>
)

data class ReportCreateDto(
    val scoreId: String,
    val type: ReportType,
    val name: String,
    val period: TimeRange,
    val lpu: String,
    val mkb: String,
    val vidmp: String,
    val medProfile: String,
    val threshold: Double,
    val ranging: RangingType,
    val present: PresentType,
    val sort: SortOrder,
    val recordCount: Int
)

data class ReportUpdateDto(
    override val id: String,
    val name: String? = null
) : BatchItem

@RestController
@RequestMapping("/api/reports")
class ReportController(
    private val aisReportService: AisReportService,
    private val batchUpdateService: BatchUpdateService
) {
    @GetMapping
    fun list(
        authToken: AuthToken,
        @RequestParam(required = false) reportName: String? = null,
        @RequestParam(required = false) scoreId: String? = null,
        @RequestParam(required = false) periodDateFrom: OffsetDateTime? = null,
        @RequestParam(required = false) periodDateTo: OffsetDateTime? = null,
        @RequestParam(required = false) createDateFrom: OffsetDateTime? = null,
        @RequestParam(required = false) createDateTo: OffsetDateTime? = null,
        @RequestParam(required = false) type: ReportType? = null,
        @RequestParam(required = false) offset: Int? = null,
        @RequestParam(required = false) limit: Int? = null
    ): ReportListDto {
        val result = aisReportService.list(authToken.hashId, ReportListCommand(
            filter = ReportFilter(
                userId = authToken.userId,
                name = reportName ?: "",
                createDate = TimeRange(createDateFrom, createDateTo),
                dataPeriod = TimeRange(periodDateFrom, periodDateTo),
                scoreId = scoreId,
                type = type ?: ReportType.UNDEFINED
            ),
            paging = PagingRequest(
                limit = limit ?: 0,
                offset = offset ?: 0
            )
        ))
        return ReportListDto(
            total = result.paging.total,
            items = result.items
        )
    }

    @PostMapping
    fun create(authToken: AuthToken, @RequestBody data: ReportCreateDto): ReportDto {
        return aisReportService.createTask(authToken.hashId, TaskReportCreateCommand(
            userId = authToken.userId,
            scoreId = data.scoreId,
            type = data.type,
            name = data.name,
            period = data.period,
            lpu = data.lpu,
            mkb = data.mkb,
            vidmp = data.vidmp,
            medProfile = data.medProfile,
            threshold = data.threshold,
            ranging = data.ranging,
            present = data.present,
            sort = data.sort,
            recordCount = data.recordCount
        ))
    }

    @GetMapping("/{reportId}")
    fun info(authToken: AuthToken, @PathVariable reportId: String): ReportDto {
        return aisReportService.info(authToken.hashId, ReportInfoCommand(reportId, authToken.userId))
    }

    @GetMapping("/{reportId}/data")
    fun data(
        authToken: AuthToken,
        @PathVariable reportId: String,
        @RequestParam sort: SortOrder
    ): List<ReportDataItem> {
        return aisReportService.data(authToken.hashId, ReportDataCommand(reportId, authToken.userId, sort)).items
    }

    @PatchMapping("/update-multiple")
    fun update(authToken: AuthToken, @RequestBody data: BatchUpdate<ReportUpdateDto>): BatchResult<ReportDto> {
        return batchUpdateService(data, { update ->
            val report = aisReportService.info(authToken.hashId, ReportInfoCommand(update.id, authToken.userId))
            val edit = ReportEditCommand(
                id = update.id,
                userId = authToken.userId,
                name = update.name ?: report.name
            )
            aisReportService.edit(authToken.hashId, edit)
            report.copy(
                name = edit.name
            )
        }, { id ->
            aisReportService.delete(authToken.hashId, ReportDeleteCommand(id, authToken.userId))
        })
    }
}