package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.entity.AuthToken
import org.springframework.web.bind.annotation.*
import java.time.OffsetDateTime

data class ReportFedListDto(
    val items: List<ReportFedDto>,
    val total: Int
)

data class ReportFedUpdateDto(
    val name: String? = null
)

data class PeriodicReportFedUpdateDto(
    val isEnabled: Boolean? = null,
    val nextDateTime: OffsetDateTime? = null,
    val fomsCode: String? = null,
    val storeDir: String? = null
)

data class ReportFedCreateDto(
    val scoreId: String,
    val isCorrection: Boolean,
    val period: TimeRange,
    val orderNum: Int
)

@RestController
@RequestMapping("/api/report-fed")
class ReportFedController(
    private val aisReportFedService: AisReportFedService
) {
    @GetMapping
    fun list(
        authToken: AuthToken,
        @RequestParam(required = false) offset: Int? = null,
        @RequestParam(required = false) limit: Int? = null
    ): ReportFedListDto {
        val result = aisReportFedService.list(authToken.hashId, ReportFedListCommand(
            filter = ReportFedFilter(),
            paging = PagingRequest(
                limit = limit ?: 0,
                offset = offset ?: 0
            )
        ))
        return ReportFedListDto(
            total = result.paging.total,
            items = result.items
        )
    }

    @PostMapping
    fun create(authToken: AuthToken, @RequestBody data: ReportFedCreateDto): ReportFedDto {
        return aisReportFedService.create(authToken.hashId, TaskReportFedCreateCommand(
            userId = authToken.userId,
            scoreId = data.scoreId,
            isCorrection = data.isCorrection,
            period = data.period,
            orderNum = data.orderNum
        ))
    }

    @GetMapping("/{reportId}")
    fun info(authToken: AuthToken, @PathVariable reportId: String): ReportFedDto {
        return aisReportFedService.info(authToken.hashId, ReportFedInfoCommand(reportId, authToken.userId))
    }

    @PatchMapping("/{reportId}")
    fun update(
        authToken: AuthToken,
        @PathVariable reportId: String,
        @RequestBody data: ReportFedUpdateDto
    ): ReportFedDto {
        var report = aisReportFedService.info(authToken.hashId, ReportFedInfoCommand(reportId, authToken.userId))
        report = report.copy(
            name = data.name ?: report.name
        )
        aisReportFedService.edit(authToken.hashId, ReportFedEditCommand(
            id = reportId,
            userId = authToken.userId,
            name = report.name
        ))
        return report
    }

    @DeleteMapping("/{reportId}")
    fun delete(authToken: AuthToken, @PathVariable reportId: String) {
        return aisReportFedService.delete(authToken.hashId, ReportFedDeleteCommand(reportId, authToken.userId))
    }

    @PostMapping("/periodic/send-now")
    fun periodicSend(authToken: AuthToken) {
        aisReportFedService.periodicSend(authToken.hashId)
    }

    @GetMapping("/periodic")
    fun periodicInfo(authToken: AuthToken): TaskReportFedDto {
        return aisReportFedService.periodicInfo(authToken.hashId)
    }

    @PatchMapping("/periodic")
    fun periodicUpdate(authToken: AuthToken, @RequestBody data: PeriodicReportFedUpdateDto): TaskReportFedDto {
        var report = aisReportFedService.periodicInfo(authToken.hashId)
        report = report.copy(
            isEnabled = data.isEnabled ?: report.isEnabled,
            nextDateTime = data.nextDateTime ?: report.nextDateTime,
            fomsCode = data.fomsCode ?: report.fomsCode,
            storeDir = data.storeDir ?: report.storeDir
        )
        aisReportFedService.periodicEdit(authToken.hashId, report)
        return report
    }

    @GetMapping("/name")
    fun getName(authToken: AuthToken): ReportFedNameDto {
        return aisReportFedService.name(authToken.hashId)
    }
}