package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.entity.AuthToken
import org.springframework.web.bind.annotation.*
import java.time.OffsetDateTime

data class ScoreListDto(
    val total: Int,
    val items: List<ScoreDto>
)

data class ScoreUpdateDto(
    override val id: String,
    val name: String?,
    val description: String?,
    val isPublic: Boolean?
) : BatchItem

@RestController
@RequestMapping("/api/scores")
class ScoreController(
    private val aisScoreService: AisScoreService,
    private val batchUpdateService: BatchUpdateService
) {
    @GetMapping
    fun list(
        authToken: AuthToken,
        @RequestParam(required = false) scoreName: String? = null,
        @RequestParam(required = false) userId: String? = null,
        @RequestParam(required = false) applyDateFrom: OffsetDateTime? = null,
        @RequestParam(required = false) applyDateTo: OffsetDateTime? = null,
        @RequestParam(required = false) createDateFrom: OffsetDateTime? = null,
        @RequestParam(required = false) createDateTo: OffsetDateTime? = null,
        @RequestParam(required = false) show: ShowFilter? = null,
        @RequestParam(required = false) otherUsers: List<String>? = null,
        @RequestParam(required = false) taskScoreId: String? = null,
        @RequestParam(required = false) modelId: String? = null,
        @RequestParam(required = false) archive: ArchiveFilter? = null,
        @RequestParam(required = false) offset: Int? = null,
        @RequestParam(required = false) limit: Int? = null
    ): ScoreListDto {
        val scores = aisScoreService.list(authToken.hashId, ScoreListCommand(
            filter = ScoreFilter(
                userId = userId ?: authToken.userId,
                name = scoreName ?: "",
                createDate = TimeRange(
                    begin = createDateFrom,
                    end = createDateTo
                ),
                dataPeriod = TimeRange(
                    begin = applyDateFrom,
                    end = applyDateTo
                ),
                show = show ?: ShowFilter.SELF,
                otherUsers = otherUsers ?: emptyList(),
                taskId = taskScoreId,
                modelId = modelId,
                archive = archive ?: ArchiveFilter.NO
            )
        )).items
        return ScoreListDto(
            total = 100,
            items = scores
        )
    }

    @GetMapping("/{scoreId}")
    fun info(authToken: AuthToken, @PathVariable scoreId: String): ScoreDto {
        return aisScoreService.info(authToken.hashId, ScoreInfoCommand(scoreId, authToken.userId))
    }

    @PostMapping("/{scoreId}/send")
    fun send(authToken: AuthToken, @PathVariable scoreId: String): ScoreDto {
        val score = aisScoreService.info(authToken.hashId, ScoreInfoCommand(scoreId, authToken.userId))
        aisScoreService.send(authToken.hashId, NeedSendScoreCommand(scoreId, score.userId))
        return aisScoreService.info(authToken.hashId, ScoreInfoCommand(scoreId, authToken.userId))
    }

    @PatchMapping("/update-multiple")
    fun update(authToken: AuthToken, @RequestBody data: BatchUpdate<ScoreUpdateDto>): BatchResult<ScoreDto> {
        return batchUpdateService(data, { update ->
            val score = aisScoreService.info(authToken.hashId, ScoreInfoCommand(update.id, authToken.userId))
            val edit = ScoreEditCommand(
                id = score.id,
                userId = score.userId,
                name = update.name ?: score.name,
                description = update.description ?: score.description,
                isPublic = update.isPublic ?: score.isPublic
            )
            aisScoreService.edit(authToken.hashId, edit)
            score.copy(
                name = edit.name,
                description = edit.description,
                isPublic = edit.isPublic
            )
        }, { id ->
            aisScoreService.delete(authToken.hashId, ScoreDeleteCommand(id, authToken.userId, data.forceDelete))
        })
    }
}