package aisexpert.backend.web

import aisexpert.backend.ais.TaskExecStatus
import aisexpert.backend.ais.TaskType
import aisexpert.backend.utils.userId
import org.slf4j.LoggerFactory
import org.springframework.messaging.simp.SimpMessageHeaderAccessor
import org.springframework.messaging.simp.SimpMessageType
import org.springframework.messaging.simp.SimpMessagingTemplate
import org.springframework.messaging.simp.user.SimpSubscription
import org.springframework.stereotype.Component
import java.time.OffsetDateTime

const val QUEUE_TASK_PROGRESS = "/queue/task/progress"

const val QUEUE_TASK_CONTENT_CREATE = "/queue/task/content-create"

const val QUEUE_SUPERVISING = "/queue/supervising"
const val USER_QUEUE_SUPERVISING = "/user/queue/supervising"

data class TrulyTaskProgress(
    val taskType: TaskType,
    val taskId: String,
    val progressCurrent: Int,
    val progressTotal: Int,
    val name: String,
    val description: String,
    val createDate: OffsetDateTime,
    val execStatus: TaskExecStatus,
    val waitPosition: Int
)

data class TrulyTaskContentCreate<C>(
    val taskType: TaskType,
    val taskId: String,
    val content: C,
    val userId: String?
)

@Component
class SimpService(private val messagingTemplate: SimpMessagingTemplate) {
    private val logger = LoggerFactory.getLogger(SimpService::class.java)

    fun sendToSupervising(message: SupervisingDto, sessions: Iterable<SimpSubscription>) {
        sendToUsers(message, QUEUE_SUPERVISING, sessions)
    }

    fun sendToTaskProgress(message: TrulyTaskProgress, sessions: Iterable<SimpSubscription>) {
        sendToUsers(message, QUEUE_TASK_PROGRESS, sessions)
    }

    fun sendToTaskCreate(message: TrulyTaskContentCreate<*>, sessions: Iterable<SimpSubscription>) {
        sendToUsers(message, QUEUE_TASK_CONTENT_CREATE, sessions)
    }

    private fun sendToUsers(message: Any, topic: String, subscriptions: Iterable<SimpSubscription>) {
        val users = subscriptions
            .map { it.session.user.name.userId }
            .distinct()

        logger.info("Sending message to users: $users")

        val sessions = subscriptions
            .map { it.session.id }
            .distinct()

        for (session in sessions) {
            val accessor = SimpMessageHeaderAccessor.create(SimpMessageType.MESSAGE)
            accessor.sessionId = session
            accessor.setLeaveMutable(true)
            messagingTemplate.convertAndSendToUser(session, topic, message, accessor.messageHeaders)
        }
    }
}
