package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.entity.AuthToken
import aisexpert.backend.spring.AisEventHandler
import aisexpert.backend.spring.AisListener
import aisexpert.backend.utils.secret
import org.slf4j.LoggerFactory
import org.springframework.messaging.simp.user.SimpUserRegistry
import org.springframework.scheduling.annotation.Scheduled
import org.springframework.web.bind.annotation.*

data class SupervisingDto(
    val hardware: SupervisingHardwareDto,
    val software: SupervisingSoftwareDto,
    val thresholds: SupervisingThresholdsDto
)

data class ThresholdsUpdateDto(
    val cpu: SupervisingThresholdDto? = null,
    val ram: SupervisingThresholdDto? = null,
    val hdd: SupervisingThresholdDto? = null,
    val cputmp: SupervisingThresholdDto? = null
)

@AisListener
@RestController
@RequestMapping("/api/supervising")
class SupervisingController(
    private val simp: SimpService,
    private val userRegistry: SimpUserRegistry,
    private val userAccessService: UserAccessService,
    private val aisSupervisingService: AisSupervisingService,
    private val serviceAuthHolder: AisServiceAuthHolder
) {
    private val logger = LoggerFactory.getLogger(SupervisingController::class.java)
    private var thresholds: SupervisingThresholdsDto? = null

    @GetMapping
    fun list(authToken: AuthToken) = list(authToken.hashId)

    @PatchMapping("/thresholds")
    fun updateThresholds(authToken: AuthToken, @RequestBody data: ThresholdsUpdateDto): SupervisingThresholdsDto {
        var thresholds = aisSupervisingService.getThresholds(authToken.hashId)
        thresholds = thresholds.copy(
            cpu = data.cpu ?: thresholds.cpu,
            ram = data.ram ?: thresholds.ram,
            hdd = data.hdd ?: thresholds.hdd,
            cputmp = data.cputmp ?: thresholds.cputmp
        )
        aisSupervisingService.editThresholds(authToken.hashId, thresholds)
        return thresholds
    }

    @Scheduled(fixedRateString = "\${web.supervising-fetch-interval}")
    fun fetchEntities() {
        val subs = userRegistry.findSubscriptions {
            userAccessService.hasSupervising(it.session.user.name.secret, ZoneRight.READ)
        }

        if (subs.isNotEmpty()) {
            logger.debug("Fetching entities...")

            val data = list(serviceAuthHolder.auth)
            simp.sendToSupervising(data, subs)
        }
    }

    @AisEventHandler("c4f71462-1b05-43b4-a9b3-ccca214b460e")
    fun thresholdsUpdated(event: SupervisingThresholdsDto) {
        logger.info("Thresholds updated $event")
        thresholds = event
    }

    private fun list(authId: Long): SupervisingDto {
        val thresholds = this.thresholds ?: aisSupervisingService.getThresholds(authId)
        this.thresholds = thresholds

        val values = aisSupervisingService.get(authId)
        return SupervisingDto(
            software = values.software.copy(
                activeUserCount = userRegistry.userCount
            ),
            hardware = values.hardware,
            thresholds = thresholds
        )
    }
}