package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.entity.AuthToken
import com.jcabi.manifests.Manifests
import org.springframework.web.bind.annotation.*

data class LogDto(
    val items: List<EventLogItem>,
    val total: Int
)

data class BackendVersionDto(
    val aisVersion: String,
    val proxyVersion: String
)

@RestController
@RequestMapping("/api/system")
class SystemController(
    private val aisEventLogService: AisEventLogService,
    private val aisVersionService: AisVersionService,
    private val aisXgbService: AisXgbService
) {
    @GetMapping("/log")
    fun get(
        authToken: AuthToken,
        @RequestParam(required = false) userId: String? = null,
        @RequestParam(required = false) taskId: String? = null,
        @RequestParam(required = false) selectAll: Boolean = false,
        @RequestParam(required = false) offset: Int? = null,
        @RequestParam(required = false) limit: Int? = null
    ): LogDto {
        val result = aisEventLogService.list(authToken.hashId, EventLogListCommand(
            filter = EventFilter(
                userId = userId,
                taskId = taskId,
                selectAll = selectAll,
                descendSort = true
            ),
            paging = PagingRequest(
                offset = offset ?: 0,
                limit = limit ?: 0
            )
        ))

        return LogDto(
            items = result.items,
            total = result.paging.total
        )
    }

    @GetMapping("/version")
    fun get(authToken: AuthToken): BackendVersionDto {
        val gitVersion = if (Manifests.exists("Git-Version")) Manifests.read("Git-Version") else null
        val gitRev =  if (Manifests.exists("Git-Revision")) Manifests.read("Git-Revision") else null
        val proxyVersion = "$gitVersion ($gitRev)"

        val aisVersion = aisVersionService.version(authToken.hashId)
        return BackendVersionDto(
            aisVersion = "${aisVersion.strver} (${aisVersion.gitrev})",
            proxyVersion =  proxyVersion
        )
    }

    @GetMapping("/xgb")
    fun info(authToken: AuthToken): XgbDto {
        return aisXgbService.info(authToken.hashId, XgbInfoCommand(null))
    }

    @PatchMapping("/xgb")
    fun update(authToken: AuthToken, @RequestBody data: List<XgbParam>) {
        aisXgbService.edit(authToken.hashId, XgbEditCommand(null, data))
    }

    @GetMapping("/xgb/{modelId}")
    fun modelInfo(authToken: AuthToken, @PathVariable modelId: String): XgbDto {
        return aisXgbService.info(authToken.hashId, XgbInfoCommand(modelId))
    }
}