package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.entity.AuthToken
import aisexpert.backend.spring.AisEventHandler
import aisexpert.backend.spring.AisListener
import aisexpert.backend.utils.secret
import aisexpert.backend.utils.userId
import org.slf4j.LoggerFactory
import org.springframework.messaging.simp.user.SimpSubscription
import org.springframework.messaging.simp.user.SimpUserRegistry
import org.springframework.web.bind.annotation.*

@RestController
@RequestMapping("/api/tasks")
@AisListener
class TaskController(
    private val simpService: SimpService,
    private val aisTaskService: AisTaskService,
    private val userRegistry: SimpUserRegistry,
    private val userAccessService: UserAccessService,
    private val aisModelService: AisModelService,
    private val aisScoreService: AisScoreService,
    private val aisReportService: AisReportService,
    private val aisReportFedService: AisReportFedService,
    private val aisNsiService: AisNsiService,
    private val aisServiceAuthHolder: AisServiceAuthHolder
) {
    private val logger = LoggerFactory.getLogger(TaskController::class.java)

    @PostMapping("/{taskId}/start-now")
    fun startNow(authToken: AuthToken, @PathVariable taskId: String) {
        aisTaskService.startNow(authToken.hashId, TaskStartNowCommand(taskId, authToken.userId))
    }

    @GetMapping("/sync-queue")
    fun getSyncQueue(authToken: AuthToken): List<SyncInfo> {
        return aisTaskService.syncProgress(authToken.hashId, SyncProgressCommand()).items
    }

    @AisEventHandler("12561ec5-e3ba-45ff-96b8-f147456afaba")
    fun onProgress(event: TaskProgress) {
        logger.info("TaskProgress event = $event")
        val subs = selectSubsWithAccess(event.taskType, event.userId, event.isPeriodic)

        simpService.sendToTaskProgress(TrulyTaskProgress(
            taskType = event.taskType,
            taskId = event.taskId,
            progressCurrent = event.current,
            progressTotal = event.total,
            name = event.taskName,
            description = event.taskDescript,
            createDate = event.taskCreateDate,
            execStatus = event.taskExecStatus,
            waitPosition = event.waitPosition
        ), subs)
    }

    @AisEventHandler("2e51292d-c37a-4dcb-9035-6f78a38569e6")
    fun onContentCreate(event: TaskContentCreate) {
        logger.info("TaskContentCreate event = $event")
        val subs = selectSubsWithAccess(event.taskType, event.userId, event.isPeriodic)
        val content: Any? = when (event.taskType) {
            TaskType.UNDEFINED -> null
            TaskType.SYNC_DATA -> null
            TaskType.SYNC_PLAN -> null
            TaskType.SEND_SCORE -> null
            TaskType.LEARN_MODEL -> aisModelService.info(aisServiceAuthHolder.auth, ModelInfoCommand(event.contentId, event.userId))
            TaskType.SCORE_CALC -> aisScoreService.info(aisServiceAuthHolder.auth, ScoreInfoCommand(event.contentId, event.userId))
            TaskType.CREATE_REPORT -> aisReportService.info(aisServiceAuthHolder.auth, ReportInfoCommand(event.contentId, event.userId))
            TaskType.CREATE_REPORT_FED -> aisReportFedService.info(aisServiceAuthHolder.auth, ReportFedInfoCommand(event.contentId, event.userId))
            TaskType.REPORT_FED -> null
            TaskType.SYNC_NSI -> aisNsiService.syncInfo(aisServiceAuthHolder.auth)
        }
        simpService.sendToTaskCreate(TrulyTaskContentCreate(
            taskType = event.taskType,
            taskId = event.taskId,
            userId = event.userId,
            content = content
        ), subs)
    }

    private fun selectSubsWithAccess(
        taskType: TaskType,
        userId: String?,
        isPeriodic: Boolean
    ): MutableSet<SimpSubscription> {
        val roleAccess: (secret: String, id: String) -> Boolean = when (taskType) {
            TaskType.UNDEFINED -> falseCheck()
            TaskType.SYNC_DATA -> adminCheck()
            TaskType.SYNC_PLAN -> falseCheck()
            TaskType.SEND_SCORE -> roleCheck(userId, isPeriodic, userAccessService::hasApply)
            TaskType.LEARN_MODEL -> roleCheck(userId, isPeriodic, userAccessService::hasLearn)
            TaskType.SCORE_CALC -> roleCheck(userId, isPeriodic, userAccessService::hasApply)
            TaskType.CREATE_REPORT -> roleCheck(userId, isPeriodic, userAccessService::hasReports)
            TaskType.CREATE_REPORT_FED -> roleCheck(userId, isPeriodic, userAccessService::hasReportsFed)
            TaskType.REPORT_FED -> roleCheck(userId, isPeriodic, userAccessService::hasReportsFed)
            TaskType.SYNC_NSI -> adminCheck()
        }

        return userRegistry.findSubscriptions {
            val simpUser = it.session.user.name
            roleAccess(simpUser.secret, simpUser.userId)
        }
    }

    private fun roleCheck(
        userId: String?,
        isPeriodic: Boolean,
        hasRole: (secret: String, right: ZoneRight) -> Boolean
    ) = { secret: String, id: String ->
        userAccessService.hasAdmin(secret) || hasRole(secret, ZoneRight.READ) && (id == userId || isPeriodic)
    }

    private fun adminCheck() = { secret: String, _: String ->
        userAccessService.hasAdmin(secret)
    }

    private fun falseCheck() = { _: String, _: String ->
        false
    }
}
