package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.entity.AuthToken
import org.springframework.web.bind.annotation.*
import java.time.OffsetDateTime

data class TaskModelUpdateDto(
    // Task
    override val id: String,
    val name: String? = null,
    val description: String? = null,
    val regularityMonth: Int? = null,
    val regularityDay: Int? = null,
    val regularityHour: Int? = null,
    val attemptLimit: Int? = null,
    val attemptInterval: Int? = null,
    val isEnabled: Boolean? = null,
    val runDateTime: OffsetDateTime? = null,
    val skipSync: Boolean? = null,

    // TaskModel
    val period: TimeRange? = null,
    val relPeriodBegin: Short? = null,
    val relPeriodDuration: Short? = null,
    val isPublic: Boolean? = null
) : BatchItem

data class TaskModelCreateDto(
    // Task
    val name: String,
    val description: String,
    val regularityMonth: Int,
    val regularityDay: Int,
    val regularityHour: Int,
    val attemptLimit: Int,
    val attemptInterval: Int,
    val isEnabled: Boolean,
    val runDateTime: OffsetDateTime,
    val skipSync: Boolean,

    // TaskModel
    val period: TimeRange,
    val relPeriodBegin: Short,
    val relPeriodDuration: Short,
    val isPublic: Boolean
)

data class SimpleTaskModelCreateDto(
    val name: String,
    val description: String,
    val period: TimeRange,
    val isPublic: Boolean,
    val skipSync: Boolean
)

data class SimpleTaskModelUpdateDto(
    val id: String,
    val name: String? = null,
    val description: String? = null,
    val period: TimeRange? = null,
    val isPublic: Boolean? = null,
    val skipSync: Boolean? = null
)

data class PeriodicTaskModelCreateDto(
    val name: String,
    val description: String,
    val isEnabled: Boolean,
    val runDateTime: OffsetDateTime,
    val periodDuration: Short,
    val isPublic: Boolean,
    val skipSync: Boolean
)

data class PeriodicTaskModelUpdateDto(
    val id: String,
    val name: String? = null,
    val description: String? = null,
    val isEnabled: Boolean? = null,
    val runDateTime: OffsetDateTime? = null,
    val nextDateTime: OffsetDateTime? = null,
    val periodDuration: Short? = null,
    val isPublic: Boolean? = null,
    val skipSync: Boolean? = null
)

@RestController
@RequestMapping("/api/tasks/models")
class TaskModelController(
    private val aisTaskModelService: AisTaskModelService,
    private val aisTaskService: AisTaskService,
    private val batchUpdateService: BatchUpdateService
) {
    @GetMapping
    fun list(
        @RequestParam(required = false) showOtherUsers: Boolean? = null,
        authToken: AuthToken
    ) : List<TaskModelDto> {
        return aisTaskModelService.list(authToken.hashId, TaskModelListCommand(
            filter = TaskFilter(
                userId = authToken.userId,
                showOtherUsers = showOtherUsers ?: false
            )
        )).items
    }

    @GetMapping("/{taskId}")
    fun get(authToken: AuthToken, @PathVariable taskId: String): TaskModelDto {
        return aisTaskModelService.info(authToken.hashId, TaskModelInfoCommand(taskId, authToken.userId))
    }

    @PostMapping("/{taskId}/interrupt")
    fun interrupt(authToken: AuthToken, @PathVariable taskId: String) {
        aisTaskService.interrupt(authToken.hashId, TaskInterruptCommand(
            taskId = taskId,
            taskType = TaskType.LEARN_MODEL,
            userId = authToken.userId
        ))
    }

    @PatchMapping("/update-multiple")
    fun update(authToken: AuthToken, @RequestBody data: BatchUpdate<TaskModelUpdateDto>): BatchResult<TaskModelDto> {
        return batchUpdateService(data, { update ->
            var task = aisTaskModelService.info(authToken.hashId, TaskModelInfoCommand(update.id, authToken.userId))
            task = task.copy(
                name = update.name ?: task.name,
                description = update.description ?: task.description,
                regularityMonth = update.regularityMonth ?: task.regularityMonth,
                regularityDay = update.regularityDay ?: task.regularityDay,
                regularityHour = update.regularityHour ?: task.regularityHour,
                attemptLimit = update.attemptLimit ?: task.attemptLimit,
                attemptInterval = update.attemptInterval ?: task.attemptInterval,
                isEnabled = update.isEnabled ?: task.isEnabled,
                period = task.period.update(update.period),
                relPeriodBegin = update.relPeriodBegin ?: task.relPeriodBegin,
                relPeriodDuration = update.relPeriodDuration ?: task.relPeriodDuration,
                isPublic = update.isPublic ?: task.isPublic,
                runDateTime = update.runDateTime ?: task.runDateTime,
                skipSync = update.skipSync ?: task.skipSync
            )
            aisTaskModelService.edit(authToken.hashId, task)
            task
        }, { id ->
            aisTaskModelService.delete(authToken.hashId, TaskModelDeleteCommand(id, authToken.userId))
        })
    }

    @PostMapping
    fun create(authToken: AuthToken, @RequestBody data: TaskModelCreateDto): TaskModelDto {
        val result = aisTaskModelService.create(authToken.hashId, TaskModelCreateCommand(
            userId = authToken.userId,
            parentId = null,
            name = data.name,
            description = data.description,
            regularityMonth = data.regularityMonth,
            regularityDay = data.regularityDay,
            regularityHour = data.regularityHour,
            attemptLimit = data.attemptLimit,
            attemptInterval = data.attemptInterval,
            isEnabled = data.isEnabled,
            period = data.period,
            relPeriodBegin = data.relPeriodBegin,
            relPeriodDuration = data.relPeriodDuration,
            isPublic = data.isPublic,
            execStatus = TaskExecStatus.NOT_RUN,
            runDateTime = data.runDateTime,
            runChildTask = false,
            skipSync = data.skipSync
        ))
        return aisTaskModelService.info(authToken.hashId, TaskModelInfoCommand(result.id, authToken.userId))
    }

    @PostMapping("/simple")
    fun createSimple(authToken: AuthToken, @RequestBody data: SimpleTaskModelCreateDto): TaskModelDto {
        val result = aisTaskModelService.create(authToken.hashId, TaskModelCreateCommand(
            userId = authToken.userId,
            parentId = null,
            name = data.name,
            description = data.description,
            regularityMonth = 0,
            regularityDay = 0,
            regularityHour = 0,
            attemptLimit = TASK_ATTEMPT_LIMIT,
            attemptInterval = TASK_ATTEMPT_INTERVAL,
            isEnabled = true,
            period = data.period,
            relPeriodBegin = 0,
            relPeriodDuration = 0,
            isPublic = data.isPublic,
            execStatus = TaskExecStatus.NOT_RUN,
            runDateTime = OffsetDateTime.now(),
            runChildTask = false,
            skipSync = data.skipSync
        ))
        return aisTaskModelService.info(authToken.hashId, TaskModelInfoCommand(result.id, authToken.userId))
    }

    @PatchMapping("/simple")
    fun updateSimple(authToken: AuthToken, @RequestBody data: SimpleTaskModelUpdateDto): TaskModelDto {
        var task = aisTaskModelService.info(authToken.hashId, TaskModelInfoCommand(data.id, authToken.userId))
        task = task.copy(
            name = data.name ?: task.name,
            description = data.description ?: task.description,
            regularityMonth = 0,
            regularityDay = 0,
            regularityHour = 0,
            period = task.period.update(data.period),
            relPeriodBegin = 0,
            relPeriodDuration = 0,
            isPublic = data.isPublic ?: task.isPublic,
            runChildTask = false,
            skipSync = data.skipSync ?: task.skipSync
        )
        aisTaskModelService.edit(authToken.hashId, task)
        return task
    }

    @PostMapping("/periodic")
    fun createPeriodic(authToken: AuthToken, @RequestBody data: PeriodicTaskModelCreateDto): TaskModelDto {
        val result = aisTaskModelService.create(authToken.hashId, TaskModelCreateCommand(
            userId = authToken.userId,
            parentId = null,
            name = data.name,
            description = data.description,
            regularityMonth = 1,
            regularityDay = 0,
            regularityHour = 0,
            attemptLimit = TASK_ATTEMPT_LIMIT,
            attemptInterval = TASK_ATTEMPT_INTERVAL,
            isEnabled = data.isEnabled,
            period = TimeRange(null, null),
            relPeriodBegin = 0,
            relPeriodDuration = data.periodDuration,
            isPublic = data.isPublic,
            execStatus = TaskExecStatus.NOT_RUN,
            runDateTime = data.runDateTime,
            runChildTask = false,
            skipSync = data.skipSync
        ))
        return aisTaskModelService.info(authToken.hashId, TaskModelInfoCommand(result.id, authToken.userId))
    }

    @PatchMapping("/periodic")
    fun updatePeriodic(authToken: AuthToken, @RequestBody data: PeriodicTaskModelUpdateDto): TaskModelDto {
        var task = aisTaskModelService.info(authToken.hashId, TaskModelInfoCommand(data.id, authToken.userId))
        task = task.copy(
            name = data.name ?: task.name,
            description = data.description ?: task.description,
            regularityMonth = 1,
            regularityDay = 0,
            regularityHour = 0,
            isEnabled = data.isEnabled ?: task.isEnabled,
            relPeriodBegin = 0,
            relPeriodDuration = data.periodDuration ?: task.relPeriodDuration,
            isPublic = data.isPublic ?: task.isPublic,
            runDateTime = data.runDateTime ?: task.runDateTime,
            nextDateTime = data.nextDateTime ?: task.nextDateTime,
            runChildTask = false,
            skipSync = data.skipSync ?: task.skipSync
        )
        aisTaskModelService.edit(authToken.hashId, task)
        return task
    }
}