package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.entity.AuthToken
import org.springframework.web.bind.annotation.*
import java.time.OffsetDateTime

data class TaskScoreUpdateDto(
    // Task
    override val id: String,
    val name: String? = null,
    val description: String? = null,
    val regularityMonth: Int? = null,
    val regularityDay: Int? = null,
    val regularityHour: Int? = null,
    val attemptLimit: Int? = null,
    val attemptInterval: Int? = null,
    val isEnabled: Boolean? = null,
    val runDateTime: OffsetDateTime? = null,
    val skipSync: Boolean? = null,

    // TaskModel
    val period: TimeRange? = null,
    val relPeriodBegin: Short? = null,
    val relPeriodDuration: Short? = null,
    val isPublic: Boolean? = null,
    val modelId: String? = null
) : BatchItem

data class TaskScoreCreateDto(
    // Task
    val name: String,
    val description: String,
    val regularityMonth: Int,
    val regularityDay: Int,
    val regularityHour: Int,
    val attemptLimit: Int,
    val attemptInterval: Int,
    val isEnabled: Boolean,
    val runDateTime: OffsetDateTime,
    val skipSync: Boolean,

    // TaskScore
    val period: TimeRange,
    val relPeriodBegin: Short,
    val relPeriodDuration: Short,
    val isPublic: Boolean,
    val modelId: String
)

data class SimpleTaskScoreCreateDto(
    val name: String,
    val description: String,
    val period: TimeRange,
    val isPublic: Boolean,
    val modelId: String,
    val skipSync: Boolean
)

data class SimpleTaskScoreUpdateDto(
    val id: String,
    val name: String? = null,
    val description: String? = null,
    val period: TimeRange? = null,
    val isPublic: Boolean? = null,
    val modelId: String? = null,
    val skipSync: Boolean? = null
)

data class PeriodicTaskScoreCreateDto(
    val name: String,
    val description: String,
    val isEnabled: Boolean,
    val runDateTime: OffsetDateTime,
    val periodDuration: Short,
    val isPublic: Boolean,
    val taskModelId: String,
    val runChildTask: Boolean,
    val skipSync: Boolean
)

data class PeriodicTaskScoreUpdateDto(
    val id: String,
    val name: String? = null,
    val description: String? = null,
    val isEnabled: Boolean? = null,
    val runDateTime: OffsetDateTime? = null,
    val nextDateTime: OffsetDateTime? = null,
    val periodDuration: Short? = null,
    val isPublic: Boolean? = null,
    val taskModelId: String? = null,
    val runChildTask: Boolean? = null,
    val skipSync: Boolean? = null
)

@RestController
@RequestMapping("/api/tasks/scores")
class TaskScoreController(
    private val aisTaskScoreService: AisTaskScoreService,
    private val aisTaskService: AisTaskService,
    private val batchUpdateService: BatchUpdateService
) {
    @GetMapping
    fun list(
        @RequestParam(required = false) taskModelId: String? = null,
        @RequestParam(required = false) modelId: String? = null,
        @RequestParam(required = false) showOtherUsers: Boolean? = null,
        authToken: AuthToken
    ) : List<TaskScoreDto> {
        return aisTaskScoreService.list(authToken.hashId, TaskScoreListCommand(
            filter = TaskFilter(
                userId = authToken.userId,
                parentId = taskModelId,
                modelId = modelId,
                showOtherUsers = showOtherUsers ?: false
            )
        )).items
    }

    @GetMapping("/{taskId}")
    fun get(authToken: AuthToken, @PathVariable taskId: String): TaskScoreDto {
        return aisTaskScoreService
            .info(authToken.hashId, TaskScoreInfoCommand(taskId, authToken.userId))
    }

    @PostMapping("/{taskId}/interrupt")
    fun interrupt(authToken: AuthToken, @PathVariable taskId: String) {
        aisTaskService.interrupt(authToken.hashId, TaskInterruptCommand(
            taskId = taskId,
            taskType = TaskType.SCORE_CALC,
            userId = authToken.userId
        ))
    }

    @PatchMapping("/update-multiple")
    fun update(authToken: AuthToken, @RequestBody data: BatchUpdate<TaskScoreUpdateDto>): BatchResult<TaskScoreDto> {
        return batchUpdateService(data, { update ->
            var task = aisTaskScoreService.info(authToken.hashId, TaskScoreInfoCommand(update.id, authToken.userId))
            task = task.copy(
                name = update.name ?: task.name,
                description = update.description ?: task.description,
                regularityMonth = update.regularityMonth ?: task.regularityMonth,
                regularityDay = update.regularityDay ?: task.regularityDay,
                regularityHour = update.regularityHour ?: task.regularityHour,
                attemptLimit = update.attemptLimit ?: task.attemptLimit,
                attemptInterval = update.attemptInterval ?: task.attemptInterval,
                isEnabled = update.isEnabled ?: task.isEnabled,
                period = task.period.update(update.period),
                relPeriodBegin = update.relPeriodBegin ?: task.relPeriodBegin,
                relPeriodDuration = update.relPeriodDuration ?: task.relPeriodDuration,
                isPublic = update.isPublic ?: task.isPublic,
                modelId = update.modelId ?: task.modelId,
                runDateTime = update.runDateTime ?: task.runDateTime,
                skipSync = update.skipSync ?: task.skipSync
            )
            aisTaskScoreService.edit(authToken.hashId, task)
            task
        }, { id ->
            aisTaskScoreService.delete(authToken.hashId, TaskScoreDeleteCommand(id, authToken.userId))
        })
    }

    @PostMapping
    fun create(authToken: AuthToken, @RequestBody data: TaskScoreCreateDto): TaskScoreDto {
        val result = aisTaskScoreService.create(authToken.hashId, TaskScoreCreateCommand(
            userId = authToken.userId,
            parentId = null,
            modelId = data.modelId,
            name = data.name,
            description = data.description,
            regularityMonth = data.regularityMonth,
            regularityDay = data.regularityDay,
            regularityHour = data.regularityHour,
            attemptLimit = data.attemptLimit,
            attemptInterval = data.attemptInterval,
            isEnabled = data.isEnabled,
            period = data.period,
            relPeriodBegin = data.relPeriodBegin,
            relPeriodDuration = data.relPeriodDuration,
            isPublic = data.isPublic,
            execStatus = TaskExecStatus.NOT_RUN,
            runDateTime = data.runDateTime,
            runChildTask = false,
            skipSync = data.skipSync
        ))
        return aisTaskScoreService
            .info(authToken.hashId, TaskScoreInfoCommand(result.id, authToken.userId))
    }

    @PostMapping("/simple")
    fun createSimple(authToken: AuthToken, @RequestBody data: SimpleTaskScoreCreateDto): TaskScoreDto {
        val result = aisTaskScoreService.create(authToken.hashId, TaskScoreCreateCommand(
            userId = authToken.userId,
            parentId = null,
            name = data.name,
            description = data.description,
            regularityMonth = 0,
            regularityDay = 0,
            regularityHour = 0,
            attemptLimit = TASK_ATTEMPT_LIMIT,
            attemptInterval = TASK_ATTEMPT_INTERVAL,
            isEnabled = true,
            period = data.period,
            relPeriodBegin = 0,
            relPeriodDuration = 0,
            isPublic = data.isPublic,
            execStatus = TaskExecStatus.NOT_RUN,
            runDateTime = OffsetDateTime.now(),
            modelId = data.modelId,
            runChildTask = false,
            skipSync = data.skipSync
        ))
        return aisTaskScoreService
            .info(authToken.hashId, TaskScoreInfoCommand(result.id, authToken.userId))
    }

    @PatchMapping("/simple")
    fun updateSimple(authToken: AuthToken, @RequestBody data: SimpleTaskScoreUpdateDto): TaskScoreDto {
        var task = aisTaskScoreService.info(authToken.hashId, TaskScoreInfoCommand(data.id, authToken.userId))
        task = task.copy(
            name = data.name ?: task.name,
            description = data.description ?: task.description,
            regularityMonth = 0,
            regularityDay = 0,
            regularityHour = 0,
            period = task.period.update(data.period),
            relPeriodBegin = 0,
            relPeriodDuration = 0,
            isPublic = data.isPublic ?: task.isPublic,
            modelId = data.modelId ?: task.modelId,
            skipSync = data.skipSync ?: task.skipSync
        )
        aisTaskScoreService.edit(authToken.hashId, task)
        return task
    }

    @PostMapping("/periodic")
    fun createPeriodic(authToken: AuthToken, @RequestBody data: PeriodicTaskScoreCreateDto): TaskScoreDto {
        val result = aisTaskScoreService.create(authToken.hashId, TaskScoreCreateCommand(
            userId = authToken.userId,
            parentId = data.taskModelId,
            name = data.name,
            description = data.description,
            regularityMonth = 1,
            regularityDay = 0,
            regularityHour = 0,
            attemptLimit = TASK_ATTEMPT_LIMIT,
            attemptInterval = TASK_ATTEMPT_INTERVAL,
            isEnabled = data.isEnabled,
            period = TimeRange(null ,null),
            relPeriodBegin = 0,
            relPeriodDuration = data.periodDuration,
            isPublic = data.isPublic,
            execStatus = TaskExecStatus.NOT_RUN,
            runDateTime = data.runDateTime,
            modelId = null,
            runChildTask = data.runChildTask,
            skipSync = data.skipSync
        ))
        return aisTaskScoreService
            .info(authToken.hashId, TaskScoreInfoCommand(result.id, authToken.userId))
    }

    @PatchMapping("/periodic")
    fun updatePeriodic(authToken: AuthToken, @RequestBody data: PeriodicTaskScoreUpdateDto): TaskScoreDto {
        var task = aisTaskScoreService.info(authToken.hashId, TaskScoreInfoCommand(data.id, authToken.userId))
        task = task.copy(
            name = data.name ?: task.name,
            description = data.description ?: task.description,
            regularityMonth = 1,
            regularityDay = 0,
            regularityHour = 0,
            isEnabled = data.isEnabled ?: task.isEnabled,
            relPeriodBegin = 0,
            relPeriodDuration = data.periodDuration ?: task.relPeriodDuration,
            isPublic = data.isPublic ?: task.isPublic,
            runDateTime = data.runDateTime ?: task.runDateTime,
            nextDateTime = data.nextDateTime ?: task.nextDateTime,
            parentId = data.taskModelId ?: task.parentId,
            runChildTask = data.runChildTask ?: task.runChildTask,
            skipSync = data.skipSync ?: task.skipSync
        )
        aisTaskScoreService.edit(authToken.hashId, task)
        return task
    }
}