package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.repository.AuthTokenRepository
import com.github.benmanes.caffeine.cache.Caffeine
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import java.time.Duration

@Service
class UserAccessService(
    private val authTokenRepository: AuthTokenRepository,
    private val aisUserService: AisUserService,
    private val aisRoleService: AisRoleService
) {
    private val logger = LoggerFactory.getLogger(UserAccessService::class.java)

    @Suppress("LeakingThis")
    private val accessCache = Caffeine.newBuilder()
        .refreshAfterWrite(Duration.ofMinutes(10))
        .build(this::fetchAccessWrapper)

    private data class Access(
        val admin: Boolean,
        val learn: ZoneRight,
        val apply: ZoneRight,
        val reports: ZoneRight,
        val reportsFed: ZoneRight,
        val supervising: ZoneRight
    )

    private val fullAccess = Access(
        admin = true,
        learn = ZoneRight.WRITE,
        apply = ZoneRight.WRITE,
        reports = ZoneRight.WRITE,
        reportsFed = ZoneRight.WRITE,
        supervising = ZoneRight.WRITE
    )

    private val emptyAccess = Access(
        admin = false,
        learn = ZoneRight.NONE,
        apply = ZoneRight.NONE,
        reports = ZoneRight.NONE,
        reportsFed = ZoneRight.NONE,
        supervising = ZoneRight.NONE
    )

    fun hasAdmin(accessToken: String): Boolean {
        return getAccess(accessToken).admin
    }

    fun hasLearn(accessToken: String, level: ZoneRight): Boolean {
        return getAccess(accessToken).learn >= level
    }

    fun hasApply(accessToken: String, level: ZoneRight): Boolean {
        return getAccess(accessToken).apply >= level
    }

    fun hasReports(accessToken: String, level: ZoneRight): Boolean {
        return getAccess(accessToken).reports >= level
    }

    fun hasReportsFed(accessToken: String, level: ZoneRight): Boolean {
        return getAccess(accessToken).reports >= level
    }

    fun hasSupervising(accessToken: String, level: ZoneRight): Boolean {
        return getAccess(accessToken).supervising >= level
    }

    private fun getAccess(accessToken: String): Access = accessCache[accessToken] ?: emptyAccess

    private fun fetchAccess(accessToken: String): Access {
        val token = authTokenRepository.findById(accessToken)
            .orElse(null)
            ?: return emptyAccess
        if (token.isAdmin == true) return fullAccess
        val userDto = aisUserService.userInfo(UserInfoCommand(token.userId))
        val groupId = userDto.groupId ?: return emptyAccess
        val groupDto = aisRoleService.info(token.hashId, RoleInfoCommand(groupId))
        return Access(
            admin = false,
            learn = groupDto.canLearnModel,
            apply = groupDto.canApplyModel,
            reports = groupDto.canRWReport,
            reportsFed = groupDto.canReportFed,
            supervising = groupDto.canMonitoring
        )
    }

    private fun fetchAccessWrapper(accessToken: String): Access {
        logger.info("Fetching access for token '$accessToken'")
        val result = fetchAccess(accessToken)
        logger.info("Access = $result")
        return result
    }
}