package aisexpert.backend.web

import aisexpert.backend.repository.AuthTokenRepository
import org.slf4j.LoggerFactory
import org.springframework.scheduling.annotation.Scheduled
import org.springframework.stereotype.Service
import org.springframework.transaction.annotation.Transactional
import java.time.OffsetDateTime
import java.util.*
import java.util.concurrent.ConcurrentHashMap
import kotlin.collections.ArrayList

@Service
class UserActivityService(
    private val authTokenRepository: AuthTokenRepository
) {
    private val logger = LoggerFactory.getLogger(UserActivityService::class.java)
    private val activeSecrets = Collections.newSetFromMap(ConcurrentHashMap<String, Boolean>())

    @Transactional
    fun updateActivityByTokenSecret(secrets: List<String>) {
        activeSecrets.addAll(secrets)
    }

    @Suppress("UNCHECKED_CAST")
    fun getLastActiveByUserId(ids: List<String>): Map<String, OffsetDateTime> {
        val tokens = authTokenRepository.findByUserIds(ids)
        return tokens
            .groupBy { t -> t.userId }
            .mapValues { u -> u.value.mapNotNull { it.lastActive } }
            .mapValues { u -> u.value.max() }
            .filterValues { it != null }
            as Map<String, OffsetDateTime>
    }

    @Scheduled(fixedRate = 60 * 1000)
    fun updateActivity() {
        val secrets = ArrayList(activeSecrets)
        activeSecrets.removeAll(secrets)

        if (secrets.isNotEmpty()) {
            logger.info("Flushing active tokens = $secrets")
            val tokens = authTokenRepository.findAllById(secrets)
            val now = OffsetDateTime.now()
            for (token in tokens) {
                token.lastActive = now
            }
            authTokenRepository.saveAll(tokens)
        }
    }
}