package aisexpert.backend.web

import aisexpert.backend.ais.*
import aisexpert.backend.entity.AuthToken
import org.springframework.web.bind.annotation.*
import java.time.OffsetDateTime

data class UserDto(
    val id: String,
    val login: String,
    val fullName: String,
    val isActive: Boolean,
    val lastActive: OffsetDateTime?,
    val groupId: String?,
    val isAdmin: Boolean,
    val isValid: Boolean
)

data class CurrentUserDto(
    val id: String,
    val login: String,
    val fullName: String,
    val isActive: Boolean,
    val lastActive: OffsetDateTime?,
    val group: GroupDto?,
    val isAdmin: Boolean,
    val isValid: Boolean
)

data class UserListDto(
    val total: Int,
    val items: List<UserDto>
)

data class UserUpdateDto(
    override val id: String,
    val isActive: Boolean? = null,
    val groupId: String? = null
) : BatchItem

@RestController
@RequestMapping("/api/users")
class UserController(
    private val aisUserService: AisUserService,
    private val aisRoleService: AisRoleService,
    private val userActivityService: UserActivityService,
    private val batchUpdateService: BatchUpdateService
) {
    @GetMapping
    fun list(
        authToken: AuthToken,
        @RequestParam(required = false) active: Boolean? = null,
        @RequestParam(required = false) fullName: String? = null,
        @RequestParam(required = false) groups: List<String>? = null,
        @RequestParam(required = false) lastActiveFrom: OffsetDateTime? = null,
        @RequestParam(required = false) lastActiveTo: OffsetDateTime? = null,
        @RequestParam(required = false) offset: Int? = null,
        @RequestParam(required = false) limit: Int? = null
    ): UserListDto {
        val users = aisUserService.list(authToken.hashId, UserListCommand(
            filterByLogin = fullName ?: ""
        ))
        val lastActive = userActivityService.getLastActiveByUserId(users.items.map { it.id })
        return UserListDto(
            total = 100,
            items = users.items.map { it.toDto(lastActive) }
        )
    }

    @GetMapping("/search-highlights")
    fun searchHighlights(
        authToken: AuthToken,
        @RequestParam fullName: String
    ): List<String> {
        return aisUserService.list(
            authToken.hashId,
            UserListCommand(
                filterByLogin = fullName,
                items = emptyList()
            )
        ).items.map { "${it.name} (${it.login})" }
    }

    @GetMapping("/{userId}/log")
    fun log(@PathVariable userId: String): String {
        return (1..64).joinToString("\n") { "[01.02.3456 78:90] $userId какая-то запись в логе" }
    }

    @PatchMapping("/update-multiple")
    fun update(authToken: AuthToken, @RequestBody data: BatchUpdate<UserUpdateDto>): BatchResult<UserDto> {
        val lastActive = userActivityService.getLastActiveByUserId(data.updates.map { it.id })

        return batchUpdateService(data, { user ->
            user.isActive?.let { isActive ->
                aisUserService.userActiveSet(authToken.hashId, UserActiveSetCommand(user.id, isActive))
            }
            user.groupId?.let { groupId ->
                aisRoleService.assign(authToken.hashId, AssignRoleCommand(groupId, user.id))
            }
            aisUserService
                .userInfo(UserInfoCommand(user.id))
                .toDto(lastActive)
        }, {
            throw UnsupportedOperationException("Cannot delete user")
        })
    }

    @GetMapping("/current")
    fun current(authToken: AuthToken): CurrentUserDto {
        val lastActive = userActivityService.getLastActiveByUserId(listOf(authToken.userId))
        val userDto = aisUserService.userInfo(UserInfoCommand(authToken.userId)).toDto(lastActive)
        val groupDto = try {
            userDto.groupId?.let {
                aisRoleService.info(authToken.hashId, RoleInfoCommand(it)).toDto()
            }
        } catch (ex: AisException) {
            null // todo group not found
        }
        return CurrentUserDto(
            id = userDto.id,
            isAdmin = userDto.isAdmin,
            isActive = userDto.isActive,
            isValid = userDto.isValid,
            login = userDto.login,
            group = groupDto,
            lastActive = userDto.lastActive,
            fullName = userDto.fullName
        )
    }
}

private fun AisUser.toDto(lastActive: Map<String, OffsetDateTime>) = UserDto(
    id = id,
    login = login,
    fullName = name,
    isActive = isActive,
    lastActive = lastActive[id],
    groupId = groupId,
    isAdmin = isAdmin,
    isValid = isValid
)