package aisexpert.backend

import aisexpert.backend.spring.TOKEN_HEADER_KEY
import aisexpert.backend.spring.TOKEN_QUERY_KEY
import aisexpert.backend.web.UserLoginRequest
import aisexpert.backend.web.UserLoginResponse
import com.fasterxml.jackson.core.type.TypeReference
import com.fasterxml.jackson.databind.ObjectMapper
import org.apache.tomcat.websocket.WsWebSocketContainer
import org.junit.Before
import org.junit.runner.RunWith
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.test.context.SpringBootTest
import org.springframework.boot.test.web.client.TestRestTemplate
import org.springframework.boot.test.web.client.postForObject
import org.springframework.boot.web.server.LocalServerPort
import org.springframework.http.client.ClientHttpRequestInterceptor
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter
import org.springframework.messaging.converter.MappingJackson2MessageConverter
import org.springframework.messaging.simp.stomp.StompFrameHandler
import org.springframework.messaging.simp.stomp.StompHeaders
import org.springframework.messaging.simp.stomp.StompSession
import org.springframework.messaging.simp.stomp.StompSessionHandlerAdapter
import org.springframework.test.context.TestPropertySource
import org.springframework.test.context.junit4.SpringRunner
import org.springframework.web.client.RestTemplate
import org.springframework.web.socket.client.standard.StandardWebSocketClient
import org.springframework.web.socket.messaging.WebSocketStompClient
import java.net.URLEncoder
import java.util.concurrent.CompletableFuture
import java.util.concurrent.TimeUnit

@RunWith(SpringRunner::class)
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@TestPropertySource("classpath:test.properties")
abstract class ApplicationTests {
    private val restTemplate = TestRestTemplate()

    @Autowired
    lateinit var objectMapper: ObjectMapper

    @LocalServerPort
    final val port = 0

    final lateinit var baseUrl: String

    @Before
    final fun init() {
        baseUrl = "http://localhost:$port/api"
    }

    fun login(user: String, password: String): UserLoginResponse {
        return restTemplate.postForObject(
            "$baseUrl/auth/login",
            UserLoginRequest(user, password, true)
        )!!
    }

    fun authorize(restTemplate: RestTemplate, user: String, password: String): UserLoginResponse {
        val response = login(user, password)
        val token = response.token

        restTemplate.interceptors = listOf(
            ClientHttpRequestInterceptor { request, body, execution ->
                request.headers[TOKEN_HEADER_KEY] = token
                execution.execute(request, body)
            }
        )

        return response
    }

    fun stompClient(): WebSocketStompClient {
        val stompClient = WebSocketStompClient(StandardWebSocketClient(WsWebSocketContainer()))
        val converter = MappingJackson2MessageConverter()
        converter.objectMapper = objectMapper
        stompClient.messageConverter = converter
        return stompClient
    }

    fun stompSession(client: WebSocketStompClient, token: String?): StompSession {
        return stompSession(client, token, object : StompSessionHandlerAdapter() {})
    }

    fun stompSession(client: WebSocketStompClient, token: String?, handler: StompSessionHandlerAdapter): StompSession {
        val tokenStr = token?.let { URLEncoder.encode(it, "UTF-8") }
        val query = "$TOKEN_QUERY_KEY=$tokenStr"

        return client
            .connect("ws://localhost:$port/api/websocket?$query", handler)
            .get(1000, TimeUnit.MILLISECONDS)
    }

    final inline fun <reified T> subscribeForOneFuture(session: StompSession, topic: String): CompletableFuture<T> {
        val future = CompletableFuture<T>()
        val sub = session.subscribe(topic, object : StompFrameHandler {
            @Suppress("UNCHECKED_CAST")
            override fun handleFrame(headers: StompHeaders, payload: Any?) {
                future.complete(payload as T)
            }
            override fun getPayloadType(headers: StompHeaders) = object : TypeReference<T>() {}.type
        })
        future.whenComplete { _, _ -> sub.unsubscribe() }
        return future
    }

    final inline fun <reified T> subscribeForOne(session: StompSession, topic: String, timeout: Long): T {
        return subscribeForOneFuture<T>(session, topic).get(timeout, TimeUnit.MILLISECONDS)
    }

    final fun restTemplate(): RestTemplate {
        val converter = MappingJackson2HttpMessageConverter(objectMapper)
        converter.setPrettyPrint(false)
        val result = RestTemplate(HttpComponentsClientHttpRequestFactory())
        result.messageConverters.removeIf { it is MappingJackson2HttpMessageConverter }
        result.messageConverters.add(converter)
        return result
    }
}