package aisexpert.backend

import aisexpert.backend.ais.NotifyAddressDto
import aisexpert.backend.ais.NotifyTriggerDto
import aisexpert.backend.web.*
import org.junit.Assert.assertEquals
import org.junit.Assert.assertNotEquals
import org.junit.Before
import org.junit.Test
import org.springframework.web.client.RestTemplate
import org.springframework.web.client.postForObject

class NotifyTests : ApplicationTests() {
    lateinit var restTemplate: RestTemplate

    @Before
    fun initNotify() {
        restTemplate = restTemplate()
        authorize(restTemplate, "user1", "111")
    }

    @Test
    fun listAddresses() {
        val items = restTemplate.getForObject<List<NotifyAddressDto>>("$baseUrl/notify/addresses")
        assertNotEquals(items, emptyList<NotifyAddressDto>())
    }

    @Test
    fun createUpdateDeleteAddress() {
        val triggers = restTemplate.getForObject<List<NotifyTriggerDto>>("$baseUrl/notify/triggers")!!

        val address = restTemplate.postForObject<NotifyAddressDto>("$baseUrl/notify/addresses", NotifyAddressCreateDto(
            name = "foo",
            email = "foo@bar.baz",
            triggers = emptyList()
        ))!!

        try {
            assertEquals(address.name, "foo")
            assertEquals(address.email, "foo@bar.baz")
            assertEquals(address.triggers, emptyList<NotifyAddressDto>())

            val address2 = restTemplate.getForObject<NotifyAddressDto>("$baseUrl/notify/addresses/${address.id}")!!
            assertEquals(address, address2)

            val newAddress = restTemplate.patchForObject<BatchResult<NotifyAddressDto>>("$baseUrl/notify/addresses/update-multiple", BatchUpdate(
                updates = listOf(
                    NotifyAddressUpdateDto(
                        id = address.id,
                        name = "foo 2",
                        email = "foo2@bar.baz",
                        triggers = listOf(triggers.first().id)
                    )
                )
            ))!!.updates.first()

            assertEquals(newAddress.name, "foo 2")
            assertEquals(newAddress.email, "foo2@bar.baz")
            assertEquals(newAddress.triggers, listOf(triggers.first().id))

            val newAddress2 = restTemplate.getForObject<NotifyAddressDto>("$baseUrl/notify/addresses/${address.id}")!!
            assertEquals(newAddress, newAddress2)
        } finally {
            restTemplate.patchForObject<Any>(
                "$baseUrl/notify/addresses/update-multiple",
                BatchUpdate<NotifyAddressUpdateDto>(deletes = listOf(address.id))
            )
        }
    }

    @Test
    fun listTriggers() {
        val items = restTemplate.getForObject<List<NotifyTriggerDto>>("$baseUrl/notify/triggers")
        assertNotEquals(items, emptyList<NotifyTriggerDto>())
    }

    @Test
    fun createUpdateDeleteTrigger() {
        val trigger = restTemplate.postForObject<NotifyTriggerDto>("$baseUrl/notify/triggers", NotifyTriggerCreateDto(
            name = "foo",
            type = "cpu",
            emissionInterval = 10,
            messageTemplate = "CPU OVERLOAD",
            subject = "CPU",
            threshold = 90
        ))!!

        try {
            assertEquals(trigger.name, "foo")
            assertEquals(trigger.type, "cpu")
            assertEquals(trigger.emissionInterval, 10)
            assertEquals(trigger.messageTemplate, "CPU OVERLOAD")
            assertEquals(trigger.subject, "CPU")
            assertEquals(trigger.threshold, 90)

            val trigger2 = restTemplate.getForObject<NotifyTriggerDto>("$baseUrl/notify/triggers/${trigger.id}")!!
            assertEquals(trigger, trigger2)

            val newTrigger = restTemplate.patchForObject<BatchResult<NotifyTriggerDto>>("$baseUrl/notify/triggers/update-multiple", BatchUpdate(
                updates = listOf(
                    NotifyTriggerUpdateDto(
                        id = trigger.id,
                        name = "foo 2",
                        type = "ram",
                        threshold = 50,
                        subject = "RAM",
                        messageTemplate = "RAM OVERLOAD",
                        emissionInterval = 20
                    )
                )
            ))!!.updates.first()

            assertEquals(newTrigger.name, "foo 2")
            assertEquals(newTrigger.type, "ram")
            assertEquals(newTrigger.emissionInterval, 20)
            assertEquals(newTrigger.messageTemplate, "RAM OVERLOAD")
            assertEquals(newTrigger.subject, "RAM")
            assertEquals(newTrigger.threshold, 50)

            val newTrigger2 = restTemplate.getForObject<NotifyTriggerDto>("$baseUrl/notify/triggers/${trigger.id}")!!
            assertEquals(newTrigger, newTrigger2)
        } finally {
            restTemplate.patchForObject<Any>(
                "$baseUrl/notify/triggers/update-multiple",
                BatchUpdate<NotifyTriggerUpdateDto>(deletes = listOf(trigger.id))
            )
        }
    }
}