package aisexpert.backend

import aisexpert.backend.ais.TaskModelDto
import aisexpert.backend.ais.TaskScoreDto
import aisexpert.backend.ais.TimeRange
import aisexpert.backend.utils.zeroOffsetDateTime
import aisexpert.backend.web.*
import org.junit.Assert.*
import org.junit.Before
import org.junit.Test
import org.springframework.web.client.RestTemplate
import org.springframework.web.client.postForObject
import java.time.OffsetDateTime

class TaskScoreTests : ApplicationTests() {
    private lateinit var restTemplate: RestTemplate
    private val userName = "user1"
    private val password = "111"

    @Before
    fun initTaskScores() {
        restTemplate = restTemplate()
        authorize(restTemplate, userName, password)
    }

    @Test
    fun testList() {
        restTemplate.getForObject<List<TaskScoreDto>>("$baseUrl/tasks/scores")
    }

    @Test
    fun testCreateUpdateDelete() {
        val model = restTemplate.getForObject<ModelListDto>("$baseUrl/models")!!.items.first()

        val data = restTemplate.postForObject<TaskScoreDto>("$baseUrl/tasks/scores", TaskScoreCreateDto(
            name = "foo",
            description = "bar",
            regularityMonth = 0,
            regularityDay = 0,
            regularityHour = 0,
            attemptLimit = 5,
            attemptInterval = 15,
            isEnabled = false,
            period = TimeRange(
                begin = OffsetDateTime.parse("2018-02-01T00:00:00+03:00"),
                end = OffsetDateTime.parse("2018-03-01T00:00:00+03:00")
            ),
            relPeriodBegin = 0,
            relPeriodDuration = 0,
            isPublic = false,
            modelId = model.id,
            runDateTime = OffsetDateTime.now(),
            skipSync = false
        ))!!

        try {
            assertEquals("foo", data.name)
            assertEquals("bar", data.description)
            assertFalse(data.isEnabled)
            assertFalse(data.isPublic)
            assertEquals(model.id, data.modelId)
            assertNotEquals(data.createDate, zeroOffsetDateTime)

            val updated = restTemplate.patchForObject<BatchResult<TaskModelDto>>("$baseUrl/tasks/scores/update-multiple", BatchUpdate(
                updates = listOf(TaskScoreUpdateDto(
                    id = data.id,
                    isPublic = true,
                    description = "new desc",
                    name = "new name"
                ))
            ))!!.updates.first()

            assertEquals(data.id, updated.id)
            assertEquals(true, updated.isPublic)
            assertEquals("new name", updated.name)
            assertEquals("new desc", updated.description)
            assertEquals(model.id, updated.modelId)

            val data2 = restTemplate.getForObject<TaskScoreDto>("$baseUrl/tasks/scores/${data.id}")!!
            assertEquals(updated.id, data2.id)
            assertEquals(updated.isPublic, data2.isPublic)
            assertEquals(updated.name, data2.name)
            assertEquals(updated.description, data2.description)
            assertEquals(updated.modelId, data.modelId)

        } finally {
            restTemplate.patchForObject<Any>("$baseUrl/tasks/scores/update-multiple", BatchUpdate<TaskScoreUpdateDto>(
                deletes = listOf(data.id)
            ))
        }
    }

    @Test
    fun testLog() {
        val task = restTemplate.getForObject<List<TaskScoreDto>>("$baseUrl/tasks/scores")!!
            .first()

        restTemplate.getForObject<LogDto>("$baseUrl/system/log?limit=10&offset=0&taskId=${task.id}")
    }
}