import * as types from './store/types'
import { showError, showSuccess } from './utils/notifications'
import { SESSION_NOT_RELIABLE } from './WebSocketListener'
import { redirectToLogin } from './utils/utils'
import { router } from './router'
import store from './store/store'
import SupervisingEntity from './api/entities/SupervisingEntity'
import { TaskExecStatus, TaskType } from './api/entities/BaseTaskEntity'
import { TaskProgressEntity } from './api/entities/TaskProgressEntity'
import { TaskContentCreateEntity } from './api/entities/TaskContentCreateEntity'
import { ReportExecStatus } from './api/entities/ReportEntity'
import { FedReportExecStatus } from './api/entities/FedReportEntity'

// todo DEFERRED ?
const LearnModelActions = {
    [TaskExecStatus.SUCCESS]: data => showSuccess(`Модель "${data.name}" успешно обучена`, 'Обучение завершено'),
    [TaskExecStatus.ATTEMPT]: data => showError(`Завершена попытка обучения модели "${data.name}"`, 'Ошибка обучения'),
    [TaskExecStatus.FAILED]: data => showError(`Не удалось произвести обучение модели "${data.name}"`, 'Ошибка обучения'),
    [TaskExecStatus.DEFERRED]: data => showSuccess(`Задача обучения модели "${data.name}" приостановлена`, 'Обучение завершено'),
    [TaskExecStatus.INTERRUPTED]: data => showSuccess(`Выполнение задачи обучения "${data.name}" успешно прервано`, 'Обучение завершено')
}

const ApplyModelActions = {
    [TaskExecStatus.SUCCESS]: data => showSuccess(`Модель "${data.name}" успешно применена`, 'Применение завершено'),
    [TaskExecStatus.ATTEMPT]: data => showError(`Завершена попытка применения модели "${data.name}"`, 'Ошибка применения'),
    [TaskExecStatus.FAILED]: data => showError(`Не удалось произвести применение модели "${data.name}"`, 'Ошибка применения'),
    [TaskExecStatus.DEFERRED]: data => showSuccess(`Задача применения модели "${data.name}" приостановлена`, 'Применение завершено'),
    [TaskExecStatus.INTERRUPTED]: data => showSuccess(`Выполнение задачи применения "${data.name}" успешно прервано`, 'Применение завершено')
}

const ReportActions = {
    [ReportExecStatus.SUCCESS]: data => showSuccess(`Отчет "${data.name}" успешно создан`, 'Отчет создан'),
    [ReportExecStatus.FAILED]: data => showError(`Создание отчета "${data.name}" завершено с ошибкой`, 'Ошибка создания отчета')
}

const FedReportActions = {
    [FedReportExecStatus.SUCCESS]: data => showSuccess(`Отчет "${data.name}" успешно создан`, 'Отчет создан'),
    [FedReportExecStatus.FAILED]: data => showError(`Создание отчета "${data.name}" завершено с ошибкой`, 'Ошибка создания отчета')
}

const NsiActions = {
    [TaskExecStatus.SUCCESS]: data => showSuccess(`Синхронизация НСИ успешно завершена`),
    [TaskExecStatus.FAILED]: data => showError(`Синхронизация НСИ завершена с ошибкой`)
}

const SyncDataActions = {
    [TaskExecStatus.SUCCESS]: data => showSuccess(`Синхронизация данных успешно завершена`),
    [TaskExecStatus.FAILED]: data => showError(`Синхронизация данных завершена с ошибкой`)
}

/**
 * Обработчики событий для клиента
 */
class AppEventActions {
    /**
     *
     * @param data {SupervisingEntity}
     */
    static async onSupervising (data) {
        await store.dispatch(`Supervising/${types.setSupervising}`, data)
    }

    /**
     * @param data {TaskProgressEntity}
     */
    static async onTaskProgress (data) {
        console.log(data)

        await store.dispatch(`SyncQueue/${types.updateSyncInfoProgressByEvent}`, data)

        switch (data.taskType) {
            case TaskType.LEARN_MODEL: {
                await store.dispatch(`ModelTasks/${types.updateModelTaskProgressByEvent}`, data)
                if (data.execStatus in LearnModelActions) LearnModelActions[data.execStatus](data)
                break
            }
            case TaskType.SCORE_CALC: {
                await store.dispatch(`ScoreTasks/${types.updateScoreTaskProgressByEvent}`, data)
                if (data.execStatus in ApplyModelActions) ApplyModelActions[data.execStatus](data)
                break
            }
            case TaskType.CREATE_REPORT: {
                await store.dispatch(`Reports/${types.updateReportProgressByEvent}`, data)
                if (data.execStatus in ReportActions) ReportActions[data.execStatus](data)
                break
            }
            case TaskType.CREATE_REPORT_FED: {
                await store.dispatch(`FedReports/${types.updateFedReportProgressByEvent}`, data)
                if (data.execStatus in FedReportActions) FedReportActions[data.execStatus](data)
                break
            }
            case TaskType.SYNC_NSI: {
                await store.dispatch(`NSI/${types.updateNsiTaskProgressByEvent}`, data)
                if (data.execStatus in NsiActions) NsiActions[data.execStatus](data)
                break
            }

            case TaskType.SYNC_DATA: {
                await store.dispatch(`SyncData/${types.updateSyncDataTaskProgressByEvent}`, data)
                if (data.execStatus in SyncDataActions) SyncDataActions[data.execStatus](data)
                break
            }
            default:
                console.warn('Event Ignored!')
        }
    }

    /**
     *
     * @param data {TaskContentCreateEntity}
     * @returns {Promise<void>}
     */
    static async onTaskCreate (data) {
        console.log(data)

        switch (data.taskType) {
            case TaskType.LEARN_MODEL:
                await store.dispatch(`Models/${types.createModelByEvent}`, data)
                break
            case TaskType.SCORE_CALC:
                await store.dispatch(`Scores/${types.createScoreByEvent}`, data)
                break
            case TaskType.CREATE_REPORT:
                await store.dispatch(`Reports/${types.createReportByEvent}`, data)
                break
            case TaskType.CREATE_REPORT_FED:
                await store.dispatch(`FedReports/${types.createFedReportByEvent}`, data)
                break
            default:
                console.warn('Event Ignored!')
        }
    }
}

/**
 * Регистрация обработчиков на конкретных событиях с вебсокетов
 * @param listener { WebSocketListener }
 * @param dispatch
 */
export class EventActions {
    static onConnect (listener) {
        let totalActions = 0
        listener.subscribe('/user/queue/supervising', async ({ body }) => {
            totalActions++
            console.log('totalActions:', totalActions)
            const supervisingEntity = new SupervisingEntity(JSON.parse(body))
            await AppEventActions.onSupervising(supervisingEntity)
        })

        listener.subscribe('/user/queue/task/progress', async ({ body }) => {
            totalActions++
            console.log('totalActions:', totalActions)
            const taskProgress = new TaskProgressEntity(JSON.parse(body))
            await AppEventActions.onTaskProgress(taskProgress)
        })

        listener.subscribe('/user/queue/task/content-create', async ({ body }) => {
            totalActions++
            console.log('totalActions:', totalActions)
            const taskContentCreate = new TaskContentCreateEntity(JSON.parse(body))
            await AppEventActions.onTaskCreate(taskContentCreate)
        })
    }

    static onWebSocketClose ({ code }) {
        console.log('Close code:', code)

        if (code === SESSION_NOT_RELIABLE) {
            console.log('code === SESSION_NOT_RELIABLE')
            // почему-то нельзя сделать deactivate
            // await dispatch(types.stopListener)
            redirectToLogin(router) // todo Noty
        }
    }
}
