import { BaseApi } from './BaseApi'
import axios from 'axios'
import { AuthTokenEntity } from './entities/AuthTokenEntity'

export class AuthApi extends BaseApi {
    getAuth () {
        return localStorage.auth ? JSON.parse(localStorage.auth) : null
    }

    emptyAuth () {
        delete localStorage.auth
        delete axios.defaults.headers.common['Authorization']
    }

    setupAuth (authData) {
        localStorage.auth = JSON.stringify(authData)
    }

    setupAxiosAuth (token) {
        axios.defaults.headers.common['Authorization'] = token
    }

    /**
     * Авторизоваться в сервисе.
     * @async
     * @param login {string}
     * @param password {string}
     * @param removeOldTokens {Boolean}
     * @returns {Promise<void>}
     */
    async login (login, password, removeOldTokens = true) {
        this.emptyAuth()

        const { data } = await this.http.post('/login', {
            login,
            password,
            removeOldTokens
        })

        const authData = {
            login,
            token: data.token
        }
        this.setupAuth(authData)
        this.setupAxiosAuth(authData.token)
    }

    /**
     * Выйти (удалив свой токен из сервиса)
     * @async
     * @returns {Promise<void>}
     */
    async logout () {
        await this.http.post('/logout')

        this.emptyAuth()
    }

    async getTokens () {
        const { data } = await this.http.get('/tokens')
        return data.map(t => new AuthTokenEntity(t))
    }

    async revokeTokens (ids) {
        await this.http.post('/tokens/revoke', {
            secrets: ids
        })
    }
}

export const authApi = new AuthApi('/api/auth')
