import { BaseApi } from './BaseApi'
import { ListResponseEntity } from './entities/ListResponseEntity'
import { FedReportEntity, FedReportTask } from './entities/FedReportEntity'
import { FedReportName } from './entities/FedReportName'

export class FedReportsApi extends BaseApi {
    Entity = FedReportEntity

    /**
     * @returns {Promise<ListResponseEntity<FedReportEntity>>}
     */
    async getFedReports ({ offset, limit }, useCache = false) {
        const { data } = await this.http.get('/', {
            params: {
                offset,
                limit
            },
            cache: useCache
        })
        const { items, total } = data

        return new ListResponseEntity(items.map(i => new this.Entity(i)), total)
    }

    /**
     * @param createFedReportDto {CreateFedReportDto}
     * @returns {Promise<FedReportEntity>}
     */
    async createFedReport (createFedReportDto) {
        const { data } = await this.http.post('/', createFedReportDto)

        return new this.Entity(data)
    }

    /**
     * @param id {string}
     * @param updateFedReportDto {UpdateFedReportDto}
     * @returns {Promise<FedReportEntity>}
     */
    async updateFedReport (id, updateFedReportDto) {
        const { data } = await this.http.patch(`/${id}`, updateFedReportDto)

        return new this.Entity(data)
    }

    /**
     * @param id {string}
     * @returns {Promise<void>}
     */
    async deleteFedReport (id) {
        await this.http.delete(`/${id}`)
    }

    /**
     * @returns {Promise<FedReportTask>}
     */
    async getFedReportsTask () {
        const { data } = await this.http.get('/periodic')
        return new FedReportTask(data)
    }

    /**
     * @param updateFedReportTaskDto {UpdateFedReportTaskDto}
     * @returns {Promise<FedReportTask>}
     */
    async updateFedReportsTask (updateFedReportTaskDto) {
        const { data } = await this.http.patch('/periodic', updateFedReportTaskDto)
        return new FedReportTask(data)
    }

    async runReportsTask () {
        await this.http.post('/periodic/send-now')
    }

    /**
     * @returns {Promise<FedReportName>}
     */
    async getName () {
        const { data } = await this.http.get('/name')
        return new FedReportName(data)
    }
}

export const fedReportsApi = new FedReportsApi('/api/report-fed')
