/* eslint-disable indent */
import { BaseEntity, fromEnum, notNull } from './BaseEntity'
import { dateFilter } from 'src/utils/filters'
import { Period } from './Periods'

export const TaskExecStatus = {
    NOT_RUN: 'NOT_RUN',
    RUNNING: 'RUNNING',
    SUCCESS: 'SUCCESS',
    ATTEMPT: 'ATTEMPT',
    FAILED: 'FAILED',
    DEFERRED: 'DEFERRED',
    INTERRUPTED: 'INTERRUPTED',
    WAIT_SYNC: 'WAIT_SYNC'
}

export const TaskStatusTranslation = {
    [TaskExecStatus.NOT_RUN]: () => 'Не начата',
    [TaskExecStatus.RUNNING]: () => 'Выполняется',
    [TaskExecStatus.SUCCESS]: () => 'Успешно завершена',
    [TaskExecStatus.ATTEMPT]: () => 'Ожидание повторного запуска',
    [TaskExecStatus.FAILED]: () => 'Завершено с ошибкой',
    [TaskExecStatus.DEFERRED]: () => 'Отложена',
    [TaskExecStatus.INTERRUPTED]: () => 'Прервана',
    [TaskExecStatus.WAIT_SYNC]: (task) => task.waitPosition === 0 ? 'Синхронизируется' : 'Ожидает синхронизации'
}

export const TaskType = {
    LEARN_MODEL: 'LEARN_MODEL',
    SCORE_CALC: 'SCORE_CALC',
    CREATE_REPORT: 'CREATE_REPORT',
    CREATE_REPORT_FED: 'CREATE_REPORT_FED',
    SYNC_NSI: 'SYNC_NSI',
    SYNC_DATA: 'SYNC_DATA'
}

export const TaskTypeTranslation = {
    [TaskType.LEARN_MODEL]: 'Обучение модели',
    [TaskType.SCORE_CALC]: 'Применение модели',
    [TaskType.CREATE_REPORT]: 'Создание отчёта',
    [TaskType.CREATE_REPORT_FED]: 'Создание ФФОМС отчёта',
    [TaskType.SYNC_NSI]: 'Синхронизация НСИ',
    [TaskType.SYNC_DATA]: 'Синхронизация данных'
}

export class BaseTaskEntity extends BaseEntity {
    constructor ({ id, userId, parentId, parentName, name, description, createDate, runDateTime,
                   regularityMonth, regularityDay, regularityHour, nextDateTime, nextRunTime,
                   attemptLimit, attemptInterval,
                   isEnabled, execStatus, isPublic, skipSync,
                   progressCurrent, progressTotal, waitPosition,
                   period, relPeriodBegin, relPeriodDuration,
                   attemptCounter, runChildTask,
                   ...other
    }) {
        // todo убрать runDateTime, nextRunTime
        super(id, other)
        this.userId = userId || null
        this.parentId = parentId
        this.parentName = parentName
        this.name = notNull(name, 'name')
        this.description = notNull(description, 'description')
        this.createDate = new Date(notNull(createDate, 'createDate'))
        this.regularityMonth = notNull(regularityMonth, 'regularityMonth')
        this.regularityDay = notNull(regularityDay, 'regularityDay')
        this.regularityHour = notNull(regularityHour, 'regularityHour')
        this.nextDateTime = new Date(notNull(nextDateTime, 'nextDateTime'))
        this.attemptLimit = notNull(attemptLimit, 'attemptLimit')
        this.attemptInterval = notNull(attemptInterval, 'attemptInterval')
        this.isEnabled = notNull(isEnabled, 'isEnabled')
        this.execStatus = fromEnum(execStatus, TaskExecStatus, true, 'execStatus')
        this.progressCurrent = notNull(progressCurrent, 'progressCurrent')
        this.progressTotal = notNull(progressTotal, 'progressTotal')
        this.waitPosition = notNull(waitPosition, 'waitPosition')
        this.attemptCounter = Boolean(notNull(attemptCounter, 'attemptCounter')) // todo можно использовать
        this.runChildTask = Boolean(notNull(runChildTask, 'runChildTask'))
        this.period = Period.fromJson(period)
        this.relPeriodBegin = notNull(relPeriodBegin, 'relPeriodBegin')
        this.relPeriodDuration = notNull(relPeriodDuration, 'relPeriodDuration')
        this.isPublic = Boolean(notNull(isPublic, 'isPublic'))
        this.skipSync = Boolean(notNull(skipSync, 'skipSync'))
    }

    /**
     * Процент прогресса, с учетом крайних ситуаций
     * @return {number}
     */
    get progress () {
        if (this.execStatus === TaskExecStatus.SUCCESS) {
            return 100
        } else if (this.execStatus === TaskExecStatus.RUNNING || (this.execStatus === TaskExecStatus.WAIT_SYNC && this.waitPosition === 0)) {
            return this.progressTotal !== 0 ? Math.round(100 * this.progressCurrent / this.progressTotal) : 0
        }

        return 0
    }

    /**
     * Подробная версия progress, напр. '1/100', когда progressCurrent и progressTotal принимают нормальные значения
     * @return {string}
     */
    get explicitProgress () {
        if (this.progressCurrent === this.progressTotal && this.progressTotal <= 0) {
            return ''
        }
        return `${this.progressCurrent}/${this.progressTotal}`
    }

    /**
     * Человеческий формат периода дата начала - дата окончания, с точностью до числа
     * @return {?string}
     */
    get periodText () {
        return this.period.begin === null || this.period.end === null
            ? null
            : dateFilter(this.period.begin) + ' - ' + dateFilter(this.period.end)
    }

    /**
     * Задача является регулярной, если хотя бы одно поле регулярности не равно 0
     * @return {boolean}
     */
    get isRegular () {
        return this.regularityMonth !== 0 || this.regularityDay !== 0 || this.regularityHour !== 0
    }
}
