import { Period } from './Periods'
import { dateFilter } from 'src/utils/filters'
import { BaseEntity, fromEnum, notNull } from './BaseEntity'

export const FedReportExecStatus = {
    UNDEFINED: 'UNDEFINED',
    CREATING: 'CREATING',
    SUCCESS: 'SUCCESS',
    FAILED: 'FAILED'
}

export const FedReportExecStatusTranslation = {
    [FedReportExecStatus.UNDEFINED]: 'Подготовка',
    [FedReportExecStatus.CREATING]: 'Формируется',
    [FedReportExecStatus.SUCCESS]: 'Готов',
    [FedReportExecStatus.FAILED]: 'Ошибка'
}

export class FedReportEntity extends BaseEntity {
    constructor ({ id, name, period, createDate, execStatus, isPeriodic, isCorrection, scoreId, scoreName, taskId, userId, xmlData, orderNum, progressCurrent, progressTotal, ...other }) {
        super(id, other)
        this.name = notNull(name, 'name')
        this.period = Period.fromJson(notNull(period, 'period'))
        this.createDate = new Date(notNull(createDate, 'createDate'))
        this.execStatus = fromEnum(execStatus, FedReportExecStatus, true, 'execStatus')
        this.isPeriodic = notNull(isPeriodic, 'isPeriodic')
        this.isCorrection = notNull(isCorrection, 'isCorrection')
        this.scoreId = notNull(scoreId, 'scoreId')
        this.scoreName = notNull(scoreName, 'scoreName')
        this.taskId = notNull(taskId, 'taskId')
        this.userId = userId || null
        this.xmlData = notNull(xmlData, 'xmlData') // todo fixme
        this.orderNum = notNull(orderNum, 'orderNum')
        this.progressCurrent = notNull(progressCurrent, 'progressCurrent')
        this.progressTotal = notNull(progressTotal, 'progressTotal')
    }

    get periodText () {
        return dateFilter(this.period.begin) + ' - ' + dateFilter(this.period.end)
    }

    /**
     * Процент прогресса, с учетом крайних ситуаций
     * @return {number}
     */
    get progress () {
        if (this.execStatus === FedReportExecStatus.SUCCESS) {
            return 100
        } else if (this.execStatus === FedReportExecStatus.CREATING) {
            return this.progressTotal !== 0 ? Math.round(100 * this.progressCurrent / this.progressTotal) : 0
        }

        return 0
    }

    /**
     * Подробная версия progress, напр. '1/100', когда progressCurrent и progressTotal принимают нормальные значения
     * @return {string}
     */
    get explicitProgress () {
        if (this.progressCurrent === this.progressTotal && this.progressTotal <= 0) {
            return ''
        }
        return `${this.progressCurrent}/${this.progressTotal}`
    }
}

// todo validation
export class FedReportTask {
    constructor ({ fomsCode, isEnabled, storeDir, nextDateTime, parentId, parentName }) {
        this.fomsCode = notNull(fomsCode, 'fomsCode')
        this.isEnabled = notNull(isEnabled, 'isEnabled')
        this.storeDir = notNull(storeDir, 'storeDir')
        this.nextDateTime = new Date(notNull(nextDateTime, 'nextDateTime'))
        this.parentId = notNull(parentId, 'parentId')
        this.parentName = notNull(parentName, 'parentName')
    }
}
