import { fromEnum, notNull } from './BaseEntity'
import { PeriodicEntity } from './PeriodicEntity'

/** Тип отчета */
export const ReportType = {
    UNDEFINED: 'UNDEFINED',
    VIDMP: 'VIDMP',
    LPU: 'LPU',
    MKB: 'MKB',
    MED_PROFILE: 'MED_PROFILE'
}

export const ReportsTypeTranslation = {
    [ReportType.VIDMP]: 'Вид медицинской помощи',
    [ReportType.LPU]: 'Медицинская организация',
    [ReportType.MKB]: 'Диагнозы',
    [ReportType.MED_PROFILE]: 'Медицинский профиль'
}

export const ReportTypeSelection = [ReportType.VIDMP, ReportType.LPU, ReportType.MKB, ReportType.MED_PROFILE]
    .map(r => ({ id: r, name: ReportsTypeTranslation[r] }))

export const PluralReportsTypeTranslation = {
    [ReportType.VIDMP]: 'Видам медицинской помощи',
    [ReportType.LPU]: 'Медицинским организациям',
    [ReportType.MKB]: 'Диагнозам',
    [ReportType.MED_PROFILE]: 'Профилям'
}

export const PluralReportTypeSelection = [ReportType.VIDMP, ReportType.LPU, ReportType.MKB, ReportType.MED_PROFILE]
    .map(r => ({ id: r, name: PluralReportsTypeTranslation[r] }))

/** Тип ранжирования */
export const SortType = {
    DESCENDING: 'DESCENDING',
    ASCENDING: 'ASCENDING'
}

export const SortTypeTranslation = {
    [SortType.DESCENDING]: 'По убыванию',
    [SortType.ASCENDING]: 'По возрастанию'
}

export const SortTypeSelection = [SortType.DESCENDING, SortType.ASCENDING]
    .map(s => ({ id: s, name: SortTypeTranslation[s] }))

/** Тип оценки */
export const RangingType = {
    MEE_RISK: 'MEE_RISK',
    EKMP_SCORES: 'EKMP_SCORES',
    MEE_SCORES: 'MEE_SCORES',
    EKMP_RISK: 'EKMP_RISK'
}

export const RangingTypeTranslation = {
    [RangingType.MEE_RISK]: 'Риск по МЭЭ',
    [RangingType.EKMP_RISK]: 'Риск по ЭКМП',
    [RangingType.EKMP_SCORES]: 'Оценка ЭКМП',
    [RangingType.MEE_SCORES]: 'Оценка МЭЭ'
}

export const RangingTypeSelection = [RangingType.MEE_RISK, RangingType.MEE_SCORES, RangingType.EKMP_RISK, RangingType.EKMP_SCORES]
    .map(r => ({ id: r, name: RangingTypeTranslation[r] }))

/** Тип диаграммы */
export const PresentType = {
    CIRCLE: 'CIRCLE',
    LINE: 'LINE',
    TABLE: 'TABLE',
    UNDEFINED: 'UNDEFINED'
}

export const PresentTypeTranslation = {
    [PresentType.CIRCLE]: 'Круговая',
    [PresentType.LINE]: 'Линейная',
    [PresentType.TABLE]: 'Таблица',
    [PresentType.UNDEFINED]: 'Неизвестно'
}

export const PresentTypeSelection = [PresentType.CIRCLE, PresentType.LINE, PresentType.TABLE]
    .map(d => ({ id: d, name: PresentTypeTranslation[d] }))

export const ReportExecStatus = {
    UNDEFINED: 'UNDEFINED',
    CREATING: 'CREATING',
    SUCCESS: 'SUCCESS',
    FAILED: 'FAILED'
}

export class ReportEntity extends PeriodicEntity {
    constructor ({ id, name, createDate, taskId, taskName, scoreId, scoreName, userId,
        threshold, recordCount, sort, type, execStatus,
        lpu, mkb, vidmp, medProfile, ranging, present, ...other }) {
        super(id, other)
        this.name = notNull(name, 'name')
        this.createDate = new Date(notNull(createDate, 'createDate'))
        this.taskId = taskId || null
        this.taskName = taskName || null
        this.scoreId = notNull(scoreId, 'scoreId')
        this.scoreName = notNull(scoreName, 'scoreName')
        this.userId = userId || null
        this.execStatus = fromEnum(execStatus, ReportExecStatus, true, 'execStatus')

        this.lpu = notNull(lpu, 'lpu')
        this.medProfile = notNull(medProfile, 'medProfile')
        this.vidmp = notNull(vidmp, 'vidmp')
        this.mkb = notNull(mkb, 'mkb')

        this.threshold = notNull(threshold, 'threshold')
        this.recordCount = notNull(recordCount, 'recordCount')
        this.ranging = fromEnum(ranging, RangingType, true, 'ranging')
        this.present = fromEnum(present, PresentType, true, 'present')
        this.sort = fromEnum(sort, SortType, true, 'sort')
        this.type = fromEnum(type, ReportType, true, 'type')
    }
}
