import { RangingTypeTranslation, PresentTypeTranslation, ReportsTypeTranslation, SortTypeTranslation } from 'src/api/entities/ReportEntity'
import { codeLegend, nameLegend } from 'src/api/enums/ReportEnums'
import { roundFilter, dateFilter, spaceAfterComma } from 'src/utils/filters'
import UserView from '../../common/UserView'
import store from 'src/store/store'

export const getReportDataTableFields = report => {
    return [
        { id: 'code', name: codeLegend[report.type] },
        { id: 'name', name: nameLegend[report.type] },
        {
            id: 'value',
            name: RangingTypeTranslation[report.ranging],
            formatter: roundFilter,
            class: 'text-nowrap'
        }
    ]
}

export const getReportInfo = (report, scoreInfo) => {
    const user = store.getters['Users/usersById'][report.userId]
    const userName = user ? `${user.fullName} (${user.login})` : 'Имя пользователя не найдено'

    return [
        { id: 'Наименование отчета', value: report.name },
        { id: 'Отчет по', value: ReportsTypeTranslation[report.type] },
        { id: 'Использованная оценка', value: scoreInfo || 'Нет данных' },
        { id: 'Период данных', value: report.periodText },
        { id: 'Дата создания', value: dateFilter(report.createDate) },
        { id: 'Автор', component: UserView, props: { userId: report.userId }, class: 'text-primary', value: userName },
        { id: 'params', separator: true, value: 'Параметры' },

        { id: 'Вид ранжирования', value: RangingTypeTranslation[report.ranging] },
        { id: 'Тип диаграммы', value: PresentTypeTranslation[report.present] },
        { id: 'Тип сортировки', value: SortTypeTranslation[report.sort] },

        { id: 'filters', separator: true, value: 'Фильтры' },

        { id: 'Перечень медицинских организаций', value: spaceAfterComma(report.lpu || '-') },
        { id: 'Перечень профилей', value: spaceAfterComma(report.medProfile || '-') },
        { id: 'Перечень видов медицинской помощи', value: spaceAfterComma(report.vidmp || '-') },
        { id: 'Перечень диагнозов', value: spaceAfterComma(report.mkb || '-') },
        { id: 'Порог', value: roundFilter(report.threshold) },
        { id: 'Количество записей', value: report.recordCount }
    ]
}

export const getReportCsv = (report, scoreInfo, reportData, separator = ';') => {
    // https://github.com/okfn/csv.js/blob/master/csv.js#L286
    const escape = value => {
        const rxNeedsQuoting = /^\s|\s$|,|"|\n/

        if (value === null) {
            // If field is null set to empty string
            value = ''
        } else if (typeof value === 'string' && rxNeedsQuoting.test(value)) {
            value = value.replace(/"/g, '""')
            // Convert string to delimited string
            value = '"' + value + '"'
        } else if (typeof value === 'number') {
            // Convert number to string
            value = value.toString(10)
        }

        return value
    }

    const rows = []

    for (const row of getReportInfo(report, scoreInfo)) {
        if (row.separator) continue
        rows.push([escape(row.id), escape(row.value)])
    }

    rows.push([])

    if (reportData.length > 0) {
        const dataFields = getReportDataTableFields(report)
        rows.push(dataFields.map(f => escape(f.name)))

        for (const dataItem of reportData) {
            rows.push(dataFields.map(f => {
                let column = dataItem[f.id]
                if (f.formatter) column = f.formatter(column)

                return escape(column)
            }))
        }
    } else {
        rows.push(['У отчета нет данных для отображения'])
    }

    // https://stackoverflow.com/a/14966131/9872697
    // https://stackoverflow.com/a/38192870/9872697
    return 'data:text/csv;charset=utf-8,\uFEFF' + rows
        .map(e => e.join(separator))
        .join('\n')
}
