import Vue from 'vue'
import Router from 'vue-router'
import { Admin, Apply, Learn, Login, Reports, Supervise } from './views'
import store from 'src/store/store'
import { Zones, ZoneValues } from './api/entities/GroupEntity'
import Home from './views/Home'
import AdminUsers from './components/admin/users/AdminUsers'
import AdminGroups from './components/admin/groups/AdminGroups'
import AdminLogs from './components/admin/logs/AdminLogs'
import AdminSettings from './components/admin/settings/AdminSettings'
import TheVersion from './components/TheVersion'
import { LogType } from './store/modules/logs'
import ReportsLocal from './components/reports/ReportsLocal'
import ReportsFederal from './components/reports/ReportsFederal'

Vue.use(Router)

export const ROUTES = {
    LOGIN: 'login',
    HOME: 'home',
    LEARN: Zones.learn,
    APPLY: Zones.apply,
    REPORTS: 'reports',
    REPORTS_LOCAL: 'reportsLocal',
    REPORTS_FEDERAL: 'reportsFederal',
    SUPERVISING: Zones.supervising,
    ADMIN: 'admin',
    ADMIN_USERS: 'adminUsers',
    ADMIN_GROUPS: 'adminGroups',
    ADMIN_LOGS: 'adminLogs',
    ADMIN_SETTINGS: 'adminSettings',
    ADMIN_ABOUT: 'adminAbout'
}

const homeOrNone = (to, from, next) => {
    if (from.name === null || from.name === ROUTES.LOGIN) {
        next({ name: ROUTES.HOME })
    } else {
        next(false)
    }
}

const authorizedGuard = (to, from, next) => {
    const { isAuthorized } = store.getters

    if (!isAuthorized) {
        console.log('Redirect to login: no auth')
        next({ name: ROUTES.LOGIN })
        return false
    }

    next()
}

const userGuard = (to, from, next) => {
    const { userZones, isAuthorized, isAdmin } = store.getters

    if (!isAuthorized) {
        console.log('Redirect to login: no auth')
        next({ name: ROUTES.LOGIN })
        return
    }

    if (!isAdmin) {
        const userAccess = userZones[to.name] || ZoneValues.NONE
        if (userAccess === ZoneValues.NONE) {
            console.log(`Access to ${to.name} denied by userGuard (User Access = ${userAccess})`)
            homeOrNone(to, from, next)
            return
        }
    }

    next()
}

const reportsGuard = (to, from, next) => {
    const { userZones, isAuthorized, isAdmin } = store.getters

    if (!isAuthorized) {
        console.log('Redirect to login: no auth')
        next({ name: ROUTES.LOGIN })
        return
    }

    if (!isAdmin) {
        const reportAccess = userZones[Zones.reports] || ZoneValues.NONE
        const reportFedAccess = userZones[Zones.reportsFed] || ZoneValues.NONE
        if (reportAccess === ZoneValues.NONE && reportFedAccess === ZoneValues.NONE) {
            console.log(`Access to ${to.name} denied by reportsGuard (Report Access = ${reportAccess}, ReportFed Access = ${reportFedAccess})`)
            homeOrNone(to, from, next)
            return
        }
    }

    next()
}

const adminGuard = (to, from, next) => {
    const { isAuthorized, isAdmin } = store.getters

    if (!(isAuthorized && isAdmin)) {
        console.log('Redirect by adminGuard')
        homeOrNone(to, from, next)
    }

    return next()
}

const globalTitle = 'АИС Эксперт'

const pages = [
    {
        path: '/login',
        name: ROUTES.LOGIN,
        component: Login,
        meta: {
            img: '/images/fon1.png'
        }
    },
    {
        path: '/',
        name: ROUTES.HOME,
        component: Home,
        meta: {
            img: '/images/fon1.png'
        },
        beforeEnter: authorizedGuard
    },
    {
        path: '/learn',
        name: ROUTES.LEARN,
        component: Learn,
        meta: {
            title: 'Обучение - ' + globalTitle,
            img: '/images/fon2.png'
        },
        beforeEnter: userGuard
    },
    {
        path: '/apply',
        name: ROUTES.APPLY,
        component: Apply,
        meta: {
            title: 'Применение - ' + globalTitle,
            img: '/images/fon3.png'
        },
        beforeEnter: userGuard
    },
    {
        path: '/reports',
        name: ROUTES.REPORTS,
        component: Reports,
        meta: {
            title: 'Отчеты - ' + globalTitle,
            img: '/images/fon4.png'
        },
        children: [
            {
                path: '',
                name: ROUTES.REPORTS_LOCAL,
                component: ReportsLocal
            },
            {
                path: 'ffoms',
                name: ROUTES.REPORTS_FEDERAL,
                component: ReportsFederal
            },
            {
                path: '*'
            }
        ],
        beforeEnter: reportsGuard
    },
    {
        path: '/supervise',
        name: ROUTES.SUPERVISING,
        component: Supervise,
        meta: {
            title: 'Мониторинг - ' + globalTitle,
            img: '/images/fon5.png'
        },
        beforeEnter: userGuard
    },
    {
        path: '/admin',
        name: ROUTES.ADMIN,
        component: Admin,
        meta: {
            title: 'Администрирование - ' + globalTitle,
            img: '/images/fon6.png'
        },
        children: [
            {
                path: 'users',
                name: ROUTES.ADMIN_USERS,
                component: AdminUsers
            },
            {
                path: 'groups',
                name: ROUTES.ADMIN_GROUPS,
                component: AdminGroups
            },
            {
                path: 'logs',
                name: ROUTES.ADMIN_LOGS,
                component: AdminLogs,
                props: { logType: LogType.System }
            },
            {
                path: 'settings',
                name: ROUTES.ADMIN_SETTINGS,
                component: AdminSettings
            },
            {
                path: 'about',
                name: ROUTES.ADMIN_ABOUT,
                component: TheVersion
            },
            {
                path: '*'
            }
        ],
        beforeEnter: adminGuard
    },
    {
        path: '*',
        redirect: { name: ROUTES.HOME }
    }
]

const routes = [
    ...pages

    // {
    //     path: '/about',
    //     name: 'about',
    //     // route level code-splitting
    //     // this generates a separate chunk (about.[hash].js) for this route
    //     // which is lazy-loaded when the route is visited.
    //     component: () => import(/* webpackChunkName: "about" */ './views/About.vue')
    // }
]

export const router = new Router({
    mode: 'history',
    base: process.env.BASE_URL,
    routes
})

router.afterEach(to => {
    const titledRoute = to.matched.find(r => r.meta.title)
    document.title = titledRoute ? titledRoute.meta.title : globalTitle
})
