import Vue from 'vue'
import * as types from '../types'
import { fedReportsApi } from 'src/api/FedReportsApi'
import { FedReportEntity, FedReportExecStatus } from 'src/api/entities/FedReportEntity'
import { TaskExecStatus } from 'src/api/entities/BaseTaskEntity'

const getDefaultState = () => {
    return {
        loading: true,
        sending: false,
        deleting: false,
        filters: {},
        reports: [],
        total: 0,
        pageNumber: 1,
        pageSize: 10,

        taskLoading: true,
        taskRunning: false,
        taskSending: false,
        task: null
    }
}

export default {
    namespaced: true,

    state: getDefaultState(),

    mutations: {
        reset: (state) => Object.assign(state, getDefaultState()),

        setLimits (state, { pageNumber, pageSize }) {
            state.pageNumber = pageNumber
            state.pageSize = pageSize
        },

        setLoading (state, value) {
            state.loading = value
        },

        setSending (state, value) {
            state.sending = value
        },

        setDeleting (state, value) {
            state.deleting = value
        },

        setFedReports (state, { items, total }) {
            state.reports = items
            state.total = total
        },

        addFedReport (state, report) {
            const idx = state.reports.findIndex(r => r.id === report.id)
            // поскольку помимо статуса у готового отчета появляется и контент, необходимо обновить сущность целиком
            if (idx > 0) {
                Vue.set(state.reports, idx, report)
            } else {
                state.reports.unshift(report)
                state.total += 1
            }
        },

        updateFedReport (state, report) {
            const idx = state.reports.findIndex(r => r.id === report.id)
            if (idx >= 0) Vue.set(state.reports, idx, report)
        },

        /**
         *
         * @param state
         * @param event {TaskProgressEntity}
         */
        updateFedReportProgress (state, event) {
            /**
             * @type {FedReportEntity}
             */
            const r = state.reports.find(r => r.taskId === event.taskId)

            // Поскольку progress приходит всем, то отчета у большинства пользователей еще нет, create приходит уже после
            if (!r) return

            r.progressTotal = event.progressTotal
            r.progressCurrent = event.progressCurrent

            // Ребята не смогли сделать у фед отчетов совместимый execStatus, поэтому маппим тут..
            if (event.execStatus in FedReportExecStatus) {
                r.execStatus = event.execStatus
            } else {
                r.execStatus = event.execStatus === TaskExecStatus.RUNNING
                    ? FedReportExecStatus.CREATING
                    : FedReportExecStatus.UNDEFINED
            }
        },

        deleteFedReports (state, ids) {
            for (const id of ids) {
                const idx = state.reports.findIndex(r => r.id === id)
                if (idx >= 0) {
                    state.reports.splice(idx, 1)
                    state.total -= 1
                }
            }
        },

        setFedReportsTaskRun (state, value) {
            state.taskRunning = value
        },

        setFedReportsTaskSending (state, value) {
            state.taskSending = value
        },

        setFedReportsTaskLoading (state, value) {
            state.taskLoading = value
        },

        setFedReportsTask (state, fedTask) {
            state.task = fedTask
        }
    },

    actions: {
        [types.reset]: ({ commit }) => commit('reset'),

        async [types.getFedReports] ({ state, commit }, { pageNumber = -1, pageSize = -1, useCache = false } = {}) {
            if (pageNumber < 0) pageNumber = state.pageNumber
            if (pageSize < 0) pageSize = state.pageSize
            commit('setLimits', { pageNumber, pageSize })

            commit('setLoading', true)
            try {
                const reports = await fedReportsApi.getFedReports({
                    ...state.filters,
                    offset: (pageNumber - 1) * pageSize,
                    limit: pageSize
                }, useCache)

                commit('setFedReports', reports)
            } finally {
                commit('setLoading', false)
            }
        },

        /**
         * @param commit
         * @param createFedReportDto {CreateFedReportDto}
         * @returns {Promise<void>}
         */
        async [types.createFedReports] ({ commit }, createFedReportDto) {
            try {
                commit('setSending', true)
                commit('addFedReport', await fedReportsApi.createFedReport(createFedReportDto))
            } finally {
                commit('setSending', false)
            }
        },

        /**
         * @param commit
         * @param data {TaskContentCreateEntity}
         * @returns {Promise<void>}
         */
        async [types.createFedReportByEvent] ({ commit }, data) {
            commit('addFedReport', new FedReportEntity(data.content))
        },

        /**
         * @param commit
         * @param id {string}
         * @param updateFedReportDto {UpdateFedReportDto}
         * @returns {Promise<void>}
         */
        async [types.updateFedReport] ({ commit }, { id, data: updateFedReportDto }) {
            try {
                commit('setSending', true)
                commit('updateFedReport', await fedReportsApi.updateFedReport(id, updateFedReportDto))
            } finally {
                commit('setSending', false)
            }
        },

        /**
         * @param commit
         * @param data {TaskProgressEntity}
         * @returns {Promise<void>}
         */
        async [types.updateFedReportProgressByEvent] ({ commit }, data) {
            commit('updateFedReportProgress', data)
        },

        /**
         * @param commit
         * @param ids {string[]}
         * @returns {Promise<void>}
         */
        async [types.deleteFedReports] ({ commit }, ids) {
            const deleted = []
            try {
                commit('setDeleting', true)
                // todo batch delete
                for (const id of ids) {
                    await fedReportsApi.deleteFedReport(id)
                    deleted.push(id)
                }
            } finally {
                commit('deleteFedReports', deleted)

                commit('setDeleting', false)
            }
        },

        async [types.getFedReportsTask] ({ commit }) {
            try {
                commit('setFedReportsTaskLoading', true)
                commit('setFedReportsTask', await fedReportsApi.getFedReportsTask())
            } finally {
                commit('setFedReportsTaskLoading', false)
            }
        },

        /**
         * @param commit
         * @param updateFedReportTaskDto {UpdateFedReportTaskDto}
         * @returns {Promise<void>}
         */
        async [types.updateFedReportsTask] ({ commit }, updateFedReportTaskDto) {
            try {
                commit('setFedReportsTaskSending', true)
                commit('setFedReportsTask', await fedReportsApi.updateFedReportsTask(updateFedReportTaskDto))
            } finally {
                commit('setFedReportsTaskSending', false)
            }
        },

        async [types.runFedReportsTask] ({ commit }) {
            try {
                commit('setFedReportsTaskRun', true)
                await fedReportsApi.runReportsTask()
            } finally {
                commit('setFedReportsTaskRun', false)
            }
        }
    }
}
