import * as types from 'src/store/types'
import { groupsApi } from 'src/api'
import Vue from 'vue'

const getDefaultState = () => {
    return {
        groups: [],
        loading: true
    }
}

export default {
    namespaced: true,

    state: getDefaultState(),

    getters: {
        groupsSelection (state) {
            return state.groups.map(
                ({ id, name }) => ({ id, name })
            )
        },

        groupsById (state) {
            const result = {}
            state.groups.forEach(g => {
                result[g.id] = g
            })
            return result
        }
    },

    mutations: {
        setGroups (state, groups) {
            state.groups = groups
        },

        addGroup (state, group) {
            state.groups.unshift(group)
        },

        updateGroup (state, group) {
            const idx = state.groups.findIndex(g => g.id === group.id)
            if (idx >= 0) Vue.set(state.groups, idx, group)
        },

        deleteGroups (state, ids) {
            for (const id of ids) {
                const idx = state.groups.findIndex(g => g.id === id)
                if (idx >= 0) state.groups.splice(idx, 1)
            }
        },

        setLoading (state, loading) {
            state.loading = loading
        },

        reset: (state) => Object.assign(state, getDefaultState())
    },

    actions: {
        [types.reset]: ({ commit }) => commit('reset'),

        async [types.getAllGroups] ({ commit }, useCache = false) {
            commit('setLoading', true)
            try {
                commit('setGroups', await groupsApi.getAll(useCache))
            } finally {
                commit('setLoading', false)
            }
        },

        async [types.createGroup] ({ commit }, group) {
            commit('addGroup', await groupsApi.create(group))
        },

        async [types.updateGroup] ({ commit }, group) {
            commit('updateGroup', await groupsApi.update(group))
        },

        async [types.deleteGroups] ({ commit }, ids) {
            const data = await groupsApi.delete(ids)

            commit('deleteGroups', data.deletes)

            return data
        }
    }
}
