import * as types from '../types'
import { authApi } from 'src/api'
import { WebSocketListener } from 'src/WebSocketListener'
import { EventActions } from 'src/EventActions'

const getDefaultState = () => {
    return {
        listener: null
    }
}

export default {
    namespaced: true,

    state: getDefaultState(),

    mutations: {
        setListener (state, listener) {
            state.listener = listener
        },

        reset: (state) => Object.assign(state, getDefaultState())
    },

    actions: {
        [types.reset] ({ commit }) {
            commit('reset')
        },
        /**
         * Инициализация прослушивания событий, подписок
         * @returns {Promise<void>}
         */
        async [types.initListener] ({ state, dispatch, commit }) {
            const authData = authApi.getAuth()
            if (!authData) return

            if (state.listener) state.listener.deactivate()

            // потому что wss://... не работает с http и на localhost
            const protocol = window.location.protocol === 'http:' ? 'ws' : 'wss'
            const url = `${protocol}://${window.location.host}/api/websocket?auth-token=${encodeURI(authData.token)}`

            const listener = new WebSocketListener(url, false)

            listener
                .on('onConnect', frame => {
                    console.log('Connected to WS', frame)
                    EventActions.onConnect(listener)
                })
                .on('onWebSocketClose', frame => {
                    console.log('WS Closed')
                    EventActions.onWebSocketClose(frame)
                })
                .on('onDisconnect', frame => {
                    console.log('Disconnected', frame)
                })
                .on('onStompError', frame => {
                    console.log('Broker reported error: ' + frame.headers['message'])
                    console.log('Additional details: ' + frame.body)
                })

            listener.activate()

            commit('setListener', Object.freeze(listener))
        },

        /**
         * Остановка прослушивания событий.
         * @returns {Promise<void>}
         */
        async [types.stopListener] ({ state }) {
            if (state.listener) state.listener.deactivate()
        }
    }
}
