import * as types from '../types'
import { systemApi } from 'src/api'

export const LogType = {
    Task: 'Task',
    System: 'System'
}

const genSection = () => {
    return {
        loading: true,
        pageSize: 1000,
        pageNumber: 1,
        items: [],
        total: 0
    }
}

const getDefaultState = () => {
    return {
        sections: {
            [LogType.Task]: genSection(),
            [LogType.System]: genSection()
        },

        systemFilterOptions: {}
    }
}

export default {
    namespaced: true,

    state: getDefaultState(),

    mutations: {
        reset: (state) => Object.assign(state, getDefaultState()),

        /**
         * @param state
         * @param logType {LogType}
         * @param value {Boolean}
         */
        setLoading (state, { logType, value }) {
            state.sections[logType].loading = value
        },

        /**
         * @param state
         * @param filterOptions {Object}
         */
        setSystemLogFilter (state, filterOptions) {
            state.systemFilterOptions = filterOptions
        },

        /**
         * @param state
         * @param logType {LogType}
         * @param log {ListResponseEntity<LogEntity>}
         * @param pageNumber {Number}
         * @param pageSize {Number}
         */
        setLog (state, { logType, log, pageNumber, pageSize }) {
            state.sections[logType] = {
                ...state.sections[logType],
                items: log.items,
                total: log.total,
                pageNumber,
                pageSize
            }
        },

        /**
         * @param state
         * @param logType {LogType}
         */
        clearLog (state, logType) {
            state.sections[logType] = genSection()
        }
    },

    actions: {
        [types.reset]: ({ commit }) => commit('reset'),

        /**
         * @param commit
         * @param logType {LogType}
         * @param filterOptions {Object}
         * @returns {Promise<void>}
         */
        async [types.applySystemLogFilters] ({ commit }, filterOptions) {
            commit('setSystemLogFilter', filterOptions)
        },

        /**
         * @param state
         * @param commit
         * @param logType {LogType}
         * @param taskId {string?}
         * @param filterOptions {Object}
         * @param pageNumber {Number}
         * @param pageSize {Number}
         * @param useCache {Boolean}
         * @returns {Promise<void>}
         */
        async [types.getLog] ({ state, commit }, { logType, taskId, pageNumber = -1, pageSize = -1, useCache = false }) {
            if (pageNumber < 0) pageNumber = state.sections[logType].pageNumber
            if (pageSize < 0) pageSize = state.sections[logType].pageSize

            const filterOptions = logType === LogType.System
                ? state.systemFilterOptions
                : { taskId, selectAll: true }

            try {
                commit('setLoading', { logType, value: true })

                const log = await systemApi.getLog({
                    ...filterOptions,
                    offset: (pageNumber - 1) * pageSize,
                    limit: pageSize
                }, useCache)

                commit('setLog', { logType, log, pageNumber, pageSize })
            } finally {
                commit('setLoading', { logType, value: false })
            }
        },

        /**
         * @param commit
         * @param logType {LogType}
         * @returns {Promise<void>}
         */
        async [types.clearLog] ({ commit }, logType) {
            commit('clearLog', logType)

            if (logType === LogType.System) {
                commit('setSystemLogFilter', {})
            }
        }
    }
}
