import * as types from 'src/store/types'
import { taskModelApi } from 'src/api/TaskModelApi'
import { taskApi } from 'src/api/TaskApi'
import Vue from 'vue'
import { insertAfter } from '../../utils/utils'

const getDefaultState = () => {
    return {
        loading: true,
        modelTasks: [],
        filters: {}
    }
}

export default {
    namespaced: true,

    state: getDefaultState(),

    getters: {
        modelTasksSelection: state => state.modelTasks.map(m => ({ id: m.id, name: m.name })),
        regularModelTasksSelection: state => state.modelTasks
            .filter(t => t.isRegular)
            .map(t => ({ id: t.id, name: t.name }))
    },

    mutations: {
        setLoading (state, loading) {
            state.loading = loading
        },

        setModelTasks (state, modelTasks) {
            state.modelTasks = modelTasks
        },

        createModelTask (state, modelTask) {
            insertAfter(state.modelTasks, modelTask, e => e.isRegular)
        },

        updateModelTask (state, modelTask) {
            const idx = state.modelTasks.findIndex(t => t.id === modelTask.id)
            if (idx >= 0) Vue.set(state.modelTasks, idx, modelTask)
        },

        deleteModelTasks (state, ids) {
            for (const id of ids) {
                const idx = state.modelTasks.findIndex(t => t.id === id)
                if (idx >= 0) state.modelTasks.splice(idx, 1)
            }
        },

        /**
         * @param state
         * @param data {TaskProgressEntity}
         */
        updateModelTaskProgressByEvent (state, data) {
            const task = state.modelTasks.find(t => t.id === data.taskId)
            if (task) {
                task.execStatus = data.execStatus
                task.progressCurrent = data.progressCurrent
                task.progressTotal = data.progressTotal
                task.waitPosition = data.waitPosition
            }
        },

        setFilters (state, filters) {
            state.filters = filters
        },

        reset: (state) => Object.assign(state, getDefaultState())
    },

    actions: {
        [types.reset]: ({ commit }) => commit('reset'),

        async [types.getModelTasks] ({ state, commit }, useCache = false) {
            commit('setLoading', true)
            try {
                commit('setModelTasks', await taskModelApi.getAll(state.filters, useCache))
            } finally {
                commit('setLoading', false)
            }
        },

        async [types.runTask] ({ dispatch, commit }, { taskId }) {
            await taskApi.runNow(taskId)
        },

        async [types.createModelTask] ({ commit }, { isRegular, data: modelTask }) {
            const data = isRegular
                ? await taskModelApi.createPeriodic(modelTask)
                : await taskModelApi.createSimple(modelTask)

            commit('createModelTask', data)
        },

        async [types.updateModelTask] ({ commit }, { isRegular, data: modelTaskDiff }) {
            const data = isRegular
                ? await taskModelApi.updatePeriodic(modelTaskDiff)
                : await taskModelApi.updateSimple(modelTaskDiff)

            commit('updateModelTask', data)
        },

        async [types.deleteModelTasks] ({ commit }, ids) {
            const data = await taskModelApi.delete(ids)

            commit('deleteModelTasks', data.deletes)

            return data
        },

        /**
         * @param commit
         * @param data {TaskProgressEntity}
         * @returns {Promise<void>}
         */
        async [types.updateModelTaskProgressByEvent] ({ commit }, data) {
            commit('updateModelTaskProgressByEvent', data)
        },

        /**
         * @param commit
         * @param id {string}
         * @returns {Promise<void>}
         */
        async [types.interruptModelTask] ({ commit }, id) {
            await taskModelApi.interrupt(id)
        },

        async [types.applyModelTasksFilters] ({ commit }, filters = {}) {
            commit('setFilters', filters)
        }
    }
}
