import * as types from 'src/store/types'
import { modelsApi } from 'src/api'
import Vue from 'vue'
import { ShowType } from 'src/api/enums/FilterEnums'
import { insertAfter } from '../../utils/utils'

const getDefaultState = () => {
    return {
        models: [],
        filters: {},
        total: 0,
        loading: true
    }
}

export default {
    namespaced: true,

    state: getDefaultState(),

    mutations: {
        setFilters (state, filters = {}) {
            state.filters = filters
        },

        setModels (state, { total, items }) {
            state.total = total
            state.models = items
        },

        createModel (state, model) {
            insertAfter(state.models, model, e => e.isPeriodic)
            state.total += 1
        },

        updateModel (state, model) {
            const idx = state.models.findIndex(m => m.id === model.id)
            if (idx >= 0) Vue.set(state.models, idx, model)
        },

        deleteModels (state, ids) {
            for (const id of ids) {
                const idx = state.models.findIndex(m => m.id === id)
                if (idx >= 0) {
                    state.models.splice(idx, 1)
                    state.total -= 1
                }
            }
        },

        setLoading (state, loading) {
            state.loading = loading
        },

        reset: (state) => Object.assign(state, getDefaultState())
    },

    actions: {
        [types.reset]: ({ commit }) => commit('reset'),

        async [types.applyModelsFilters] ({ commit }, filters = {}) {
            commit('setFilters', filters)
        },

        async [types.getModels] ({ state, commit }, useCache = false) {
            commit('setLoading', true)
            try {
                const models = await modelsApi.getModels({
                    show: ShowType.ALL,
                    ...state.filters
                }, useCache)
                commit('setModels', models)
            } finally {
                commit('setLoading', false)
            }
        },

        /**
         *
         * @param commit
         * @param event {TaskContentCreateEntity}
         * @returns {Promise<void>}
         */
        async [types.createModelByEvent] ({ commit }, event) {
            commit('createModel', event.content)
        },

        async [types.updateModel] ({ commit }, model) {
            const newModel = await modelsApi.update(model)
            commit('updateModel', newModel)
        },

        async [types.deleteModels] ({ commit }, { ids, force = false }) {
            const data = await modelsApi.delete(ids, force)

            commit('deleteModels', data.deletes)

            return data
        }
    }
}
