import * as types from '../types'
import Vue from 'vue'
import { notifyAddressesApi } from '../../api/NotifyAddressesApi'
import { notifyTriggersApi } from '../../api/NotifyTriggersApi'

const getDefaultState = () => {
    return {
        triggersLoading: true,
        triggers: [],

        addressesLoading: true,
        addresses: [],

        variables: [
            { name: 'RECIPIENT', description: 'Получатель' },
            { name: 'USER', description: 'Пользователь' },
            { name: 'TASK', description: 'Задача' },
            { name: 'EVENT_NAME', description: 'Имя события' },
            { name: 'EVENT_DESCRIPTION', description: 'Описание события' }
        ]
    }
}

export default {
    namespaced: true,

    state: getDefaultState(),

    getters: {
        triggersById (state) {
            let triggersById = {}
            for (let trigger of state.triggers) {
                triggersById[trigger.id] = trigger
            }
            return triggersById
        }
    },

    mutations: {
        reset: (state) => Object.assign(state, getDefaultState()),

        /**
         * @param state
         * @param value {boolean}
         */
        setNotifyAddressesLoading (state, value) {
            state.triggersLoading = value
        },

        /**
         * @param state
         * @param addresses {NotifyAddressEntity[]}
         */
        setNotifyAddresses (state, addresses) {
            state.addresses = addresses
        },

        /**
         * @param state
         * @param address {NotifyAddressEntity}
         */
        createNotifyAddress (state, address) {
            state.addresses.unshift(address)
        },

        /**
         * @param state
         * @param address {NotifyAddressEntity}
         */
        updateNotifyAddress (state, address) {
            const idx = state.addresses.findIndex(a => a.id === address.id)
            if (idx >= 0) Vue.set(state.addresses, idx, address)
        },

        /**
         * @param state
         * @param ids {string[]}
         */
        deleteNotifyAddresses (state, ids) {
            state.addresses = state.addresses.filter(t => !ids.includes(t.id))
        },

        /**
         * @param state
         * @param value {boolean}
         */
        setNotifyTriggersLoading (state, value) {
            state.addressesLoading = value
        },

        /**
         * @param state
         * @param data {NotifyTriggerEntity[]}
         */
        setNotifyTriggers (state, data) {
            state.triggers = data
        },

        /**
         * @param state
         * @param trigger {NotifyTriggerEntity}
         */
        createNotifyTrigger (state, trigger) {
            state.triggers.unshift(trigger)
        },

        /**
         * @param state
         * @param trigger {NotifyTriggerEntity}
         */
        updateNotifyTrigger (state, trigger) {
            const idx = state.triggers.findIndex(t => t.id === trigger.id)
            if (idx >= 0) Vue.set(state.triggers, idx, trigger)
        },

        /**
         * @param state
         * @param ids {string[]}
         */
        deleteNotifyTriggers (state, ids) {
            state.triggers = state.triggers.filter(t => !ids.includes(t.id))

            for (let address of state.addresses) {
                address.triggers = address.triggers.filter(t => !ids.includes(t.id))
            }
        }
    },

    actions: {
        [types.reset]: ({ commit }) => commit('reset'),

        async [types.getNotifyAddresses] ({ commit }, useCache = false) {
            commit('setNotifyAddressesLoading', true)
            try {
                commit('setNotifyAddresses', await notifyAddressesApi.getAddresses(useCache))
            } finally {
                commit('setNotifyAddressesLoading', false)
            }
        },

        /**
         * @param commit
         * @param createAddressDto {CreateAddressDto}
         * @returns {Promise<void>}
         */
        async [types.createNotifyAddress] ({ commit }, createAddressDto) {
            const data = await notifyAddressesApi.createAddress(createAddressDto)
            commit('createNotifyAddress', data)
        },

        /**
         * @param commit
         * @param id {string}
         * @param updateAddressDto {UpdateAddressDto}
         * @returns {Promise<void>}
         */
        async [types.updateNotifyAddress] ({ commit }, { id, data: updateAddressDto }) {
            const data = await notifyAddressesApi.update({ id, ...updateAddressDto })
            commit('updateNotifyAddress', data)
        },

        /**
         * @param commit
         * @param id {string}
         * @returns {Promise<void>}
         */
        async [types.deleteNotifyAddresses] ({ commit }, ids) {
            const result = await notifyAddressesApi.delete(ids)
            commit('deleteNotifyAddresses', result.deletes)
            return result
        },

        /**
         * @param commit
         * @param useCache {boolean}
         * @returns {Promise<void>}
         */
        async [types.getNotifyTriggers] ({ commit }, useCache = false) {
            commit('setNotifyTriggersLoading', true)
            try {
                commit('setNotifyTriggers', await notifyTriggersApi.getTriggers(useCache))
            } finally {
                commit('setNotifyTriggersLoading', false)
            }
        },

        /**
         * @param commit
         * @param createTriggerDto {CreateTriggerDto}
         * @returns {Promise<void>}
         */
        async [types.createNotifyTrigger] ({ commit }, createTriggerDto) {
            const data = await notifyTriggersApi.createTrigger(createTriggerDto)
            commit('createNotifyTrigger', data)
        },

        /**
         * @param commit
         * @param id {string}
         * @param updateTriggerDto {UpdateTriggerDto}
         * @returns {Promise<void>}
         */
        async [types.updateNotifyTrigger] ({ commit }, { id, data: updateTriggerDto }) {
            const data = await notifyTriggersApi.update({ id, ...updateTriggerDto })
            commit('updateNotifyTrigger', data)
        },

        /**
         * @param commit
         * @param id {string}
         * @returns {Promise<void>}
         */
        async [types.deleteNotifyTriggers] ({ commit }, ids) {
            const result = await notifyTriggersApi.delete(ids)
            commit('deleteNotifyTriggers', result.deletes)
            return result
        }
    }
}
