import * as types from '../types'
import { nsiApi } from 'src/api'
import { TaskExecStatus } from '../../api/entities/BaseTaskEntity'

const getDefaultState = () => {
    return {
        scoreId: null,

        lpu: [],
        lpuLoading: false,

        mkb: [],
        mkbLoading: false,

        vidmpLoading: false,
        vidmp: [],

        medProfile: [],
        medProfileLoading: false,

        task: null,
        taskLoading: false,
        taskUpdating: false
    }
}

export default {
    namespaced: true,

    state: getDefaultState(),

    mutations: {
        reset: (state) => Object.assign(state, getDefaultState()),

        /**
         * @param state
         * @param scoreId {string}
         */
        setScoreId (state, scoreId) {
            state.scoreId = scoreId
        },

        /**
         * @param state
         * @param value {Boolean}
         */
        setLpuLoading (state, value) {
            state.lpuLoading = value
        },

        /**
         * @param state
         * @param lpu {NsiLpu[]}
         */
        setLpu (state, lpu) {
            state.lpu = lpu
        },

        /**
         * @param state
         * @param value {Boolean}
         */
        setMkbLoading (state, value) {
            state.mkbLoading = value
        },

        /**
         * @param state
         * @param mkb {NsiMkb[]}
         */
        setMkb (state, mkb) {
            state.mkb = mkb
        },

        /**
         * @param state
         * @param value {Boolean}
         */
        setMedProfileLoading (state, value) {
            state.medProfileLoading = value
        },

        /**
         * @param state
         * @param vidmp {NsiVidmp[]}
         */
        setVidmp (state, vidmp) {
            state.vidmp = vidmp
        },

        /**
         * @param state
         * @param value {Boolean}
         */
        setVidmpLoading (state, value) {
            state.vidmpLoading = value
        },

        /**
         * @param state
         * @param medProfile {NsiMedProfile[]}
         */
        setMedProfile (state, medProfile) {
            state.medProfile = medProfile
        },

        /**
         * @param state
         * @param task {NsiTaskEntity}
         */
        setTask (state, task) {
            state.task = task
        },

        /**
         * @param state
         * @param status {TaskExecStatus}
         */
        setTaskStatus (state, status) {
            state.task.inProgress = status === TaskExecStatus.RUNNING
        },

        /**
         * @param state
         * @param value {boolean}
         */
        setTaskLoading (state, value) {
            state.taskLoading = value
        },

        /**
         * @param state
         * @param value {boolean}
         */
        setTaskUpdating (state, value) {
            state.taskUpdating = value
        }
    },

    actions: {
        [types.reset]: ({ commit }) => commit('reset'),

        /**
         * @param commit
         * @param dispatch
         * @param scoreId {string}
         * @return {Promise<void>}
         */
        async [types.applyNsiScoreId] ({ commit, dispatch }, scoreId) {
            commit('setScoreId', scoreId)

            if (scoreId) {
                await Promise.all([
                    dispatch(types.getNsiLpu),
                    dispatch(types.getNsiMkb),
                    dispatch(types.getNsiVidmp),
                    dispatch(types.getNsiMedProfile)
                ])
            } else {
                commit('setLpu', [])
                commit('setMkb', [])
                commit('setVidmp', [])
                commit('setMedProfile', [])
            }
        },

        async [types.getNsiLpu] ({ state, commit }, filterOptions) {
            try {
                commit('setLpuLoading', true)
                commit('setLpu', await nsiApi.getLpu({
                    ...filterOptions,
                    scoreId: state.scoreId
                }))
            } finally {
                commit('setLpuLoading', false)
            }
        },

        async [types.getNsiMkb] ({ state, commit }, filterOptions) {
            try {
                commit('setMkbLoading', true)
                commit('setMkb', await nsiApi.getMkb({
                    ...filterOptions,
                    scoreId: state.scoreId
                }))
            } finally {
                commit('setMkbLoading', false)
            }
        },

        async [types.getNsiVidmp] ({ state, commit }) {
            try {
                commit('setVidmpLoading', true)
                commit('setVidmp', await nsiApi.getVidmp({ scoreId: state.scoreId }))
            } finally {
                commit('setVidmpLoading', false)
            }
        },

        async [types.getNsiMedProfile] ({ state, commit }) {
            try {
                commit('setMedProfileLoading', true)
                commit('setMedProfile', await nsiApi.getMedProfile({ scoreId: state.scoreId }))
            } finally {
                commit('setMedProfileLoading', false)
            }
        },

        async [types.getNsiTask] ({ commit }, useCache) {
            try {
                commit('setTaskLoading', true)
                commit('setTask', await nsiApi.getNsiTask(useCache))
            } finally {
                commit('setTaskLoading', false)
            }
        },

        /**
         * @param commit
         * @param data {UpdateNsiTaskDto}
         * @return {Promise<void>}
         */
        async [types.updateNsiTask] ({ commit }, data) {
            try {
                commit('setTaskUpdating', true)
                commit('setTask', await nsiApi.updateNsiTask(data))
            } finally {
                commit('setTaskUpdating', false)
            }
        },

        async [types.startNsiTaskNow] () {
            await nsiApi.startNsiTaskNow()
        },

        /**
         * @param commit
         * @param state
         * @param data {TaskProgressEntity}
         * @return {Promise<void>}
         */
        async [types.updateNsiTaskProgressByEvent] ({ commit, state }, data) {
            commit('setTaskStatus', data.execStatus)
        }
    }
}
