import * as types from '../types'
import { reportsApi } from 'src/api'
import Vue from 'vue'
import { ReportEntity, ReportExecStatus } from 'src/api/entities/ReportEntity'
import { TaskExecStatus } from 'src/api/entities/BaseTaskEntity'

const getDefaultState = () => {
    return {
        loading: true,
        filters: {},
        reports: [],
        total: 0,
        pageNumber: 1,
        pageSize: 10,

        reportsData: {},
        reportsDataLoading: true
    }
}

export default {
    namespaced: true,

    state: getDefaultState(),

    mutations: {
        setLoading (state, loading) {
            state.loading = loading
        },

        addReport (state, report) {
            if (!state.reports.find(r => r.id === report.id)) {
                state.reports.unshift(report)
                state.total += 1
            }
        },

        updateReport (state, report) {
            const idx = state.reports.findIndex(r => r.id === report.id)
            if (idx >= 0) Vue.set(state.reports, idx, report)
        },

        /**
         *
         * @param state
         * @param event {TaskProgressEntity}
         */
        updateReportProgress (state, event) {
            /**
             * @type {ReportEntity}
             */
            const r = state.reports.find(r => r.taskId === event.taskId)

            // Поскольку progress приходит всем, то отчета у большинства пользователей еще нет, create приходит уже после
            if (!r) return

            // fixme дублирование кода с FedReports
            // Ребята не смогли сделать у фед отчетов совместимый execStatus, поэтому маппим тут..
            if (event.execStatus in ReportExecStatus) {
                r.execStatus = event.execStatus
            } else {
                r.execStatus = event.execStatus === TaskExecStatus.RUNNING
                    ? ReportExecStatus.CREATING
                    : ReportExecStatus.UNDEFINED
            }
        },

        setFilters (state, filters) {
            state.filters = filters
        },

        setLimits (state, { pageNumber, pageSize }) {
            state.pageNumber = pageNumber
            state.pageSize = pageSize
        },

        setReports (state, { total, items }) {
            state.total = total
            state.reports = items
        },

        setReportsDataLoading (state, value) {
            state.reportsDataLoading = value
        },

        addReportData (state, { id, data }) {
            Vue.set(state.reportsData, id, data)
        },

        clearReportData (state, id) {
            Vue.delete(state.reportsData, id)
        },

        deleteReports (state, ids) {
            for (const id of ids) {
                const idx = state.reports.findIndex(r => r.id === id)
                if (idx >= 0) {
                    state.reports.splice(idx, 1)
                    state.total -= 1
                }
            }
        },

        reset: (state) => Object.assign(state, getDefaultState())
    },

    actions: {
        async [types.applyReportsFilters] ({ commit }, filters = {}) {
            commit('setFilters', filters)
        },

        async [types.getReports] ({ state, commit }, { pageNumber = -1, pageSize = -1, useCache = false }) {
            if (pageNumber < 0) pageNumber = state.pageNumber
            if (pageSize < 0) pageSize = state.pageSize
            commit('setLimits', { pageNumber, pageSize })

            commit('setLoading', true)
            try {
                const reports = await reportsApi.getAll({
                    ...state.filters,
                    offset: (pageNumber - 1) * pageSize,
                    limit: pageSize
                }, useCache)

                commit('setReports', reports)
            } finally {
                commit('setLoading', false)
            }
        },

        async [types.getReportData] ({ commit }, { id, sort }) {
            try {
                commit('setReportsDataLoading', true)
                commit('addReportData', {
                    id,
                    data: await reportsApi.getData(id, sort)
                })
            } finally {
                commit('setReportsDataLoading', false)
            }
        },

        async [types.clearReportData] ({ commit }, id) {
            commit('clearReportData', id)
            commit('setReportsDataLoading', true)
        },

        async [types.createReport] ({ commit }, report) {
            commit('addReport', await reportsApi.create(report))
        },

        /**
         * @param commit
         * @param event {TaskContentCreateEntity}
         * @returns {Promise<void>}
         */
        async [types.createReportByEvent] ({ commit }, event) {
            commit('addReport', new ReportEntity(event.content))
        },

        async [types.updateReport] ({ commit }, reportDiff) {
            commit('updateReport', await reportsApi.update(reportDiff))
        },

        /**
         * @param commit
         * @param event {TaskProgressEntity}
         * @returns {Promise<void>}
         */
        async [types.updateReportProgressByEvent] ({ commit }, event) {
            commit('updateReportProgress', event)
        },

        async [types.deleteReports] ({ commit }, ids) {
            const data = await reportsApi.delete(ids)

            commit('deleteReports', data.deletes)

            return data
        },

        [types.reset]: ({ commit }) => commit('reset')
    }
}
